DROP TABLE IF EXISTS order_products;
DROP TABLE IF EXISTS orders;
DROP TABLE IF EXISTS customers;
DROP TABLE IF EXISTS products;
DROP SEQUENCE IF EXISTS products_seq;
DROP SEQUENCE IF EXISTS orders_seq;
DROP TYPE IF EXISTS order_status;


CREATE SEQUENCE products_seq START 10;
CREATE SEQUENCE orders_seq START 10;

CREATE TABLE products (
	product_id INTEGER DEFAULT nextval('products_seq'),
	product_name VARCHAR(100) NOT NULL,
	-- 10 cyfr, z czego 2 po prawej stronie kropki
	-- zakres wartości od -99999999.99 do +99999999.99
	-- inna nazwa: DECIMAL, w Oraclu nazwa NUMBER
	price NUMERIC(10, 2) NOT NULL,
	vat NUMERIC(2, 2), -- wartości do 0.99
	description TEXT, -- w Oraclu pod nazwą CLOB
	
	PRIMARY KEY(product_id),
	CHECK (price > 0),
	CHECK (vat >= 0),
	CHECK (length(product_name) > 0)
);

-- Nie zawsze kluczem głównym jest automatycznie generowany numer.
-- Kluczem może być też kolumna tekstowa, albo typu "data z czasem" (timestamp).
-- Przykładowo tutaj napiszemy, że kluczem klienta jest jego email.
CREATE TABLE customers (
	customer_email VARCHAR(100),
	customer_name VARCHAR(100) NOT NULL,
	phone_number VARCHAR(20),
	address VARCHAR(250),
	postal_code VARCHAR(10),
	city VARCHAR(100),
	
	PRIMARY KEY(customer_email)
);

CREATE TYPE order_status AS ENUM (
	'NEW',
	'CONFIRMED',
	'PAID',
	'SHIPPED',
	'DELIVERED',
	'CLOSED',
	'RETURNED'
);

CREATE TABLE orders (
	order_id INTEGER DEFAULT nextval('orders_seq') ,
	customer_email VARCHAR(100) NOT NULL,
	status order_status DEFAULT 'NEW' NOT NULL,
	order_date TIMESTAMP DEFAULT current_timestamp NOT NULL,
	delivery_date DATE,
	
	PRIMARY KEY(order_id),
	FOREIGN KEY(customer_email) REFERENCES customers(customer_email),
	-- kolumna customer_email z tej tabeli wskazuje na kolumnę customer_email z tabeli customers
	-- (można wpisać tylko email istniejącego klienta)
	CHECK (delivery_date >= order_date::DATE)
);

CREATE TABLE order_products (
	order_id INTEGER NOT NULL,
	product_id INTEGER NOT NULL,
	quantity SMALLINT DEFAULT 1 NOT NULL,
	actual_price NUMERIC(10, 2) NOT NULL,
	actual_vat NUMERIC(2, 2),
	
	PRIMARY KEY(order_id, product_id),
	FOREIGN KEY(order_id) REFERENCES orders(order_id),
	FOREIGN KEY(product_id) REFERENCES products(product_id)
);




