package com.example.demo;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ResponseBody;

@Controller
public class TimeController {
	// Spring MVC
	
    @GetMapping("/time1")
    @ResponseBody
    public LocalTime time1() {
        // Gdy metoda zwraca jakiś obiekt (klasy innej niż String i niektóre inne "magiczne" klasy samego Springa),
        // a metoda posiada adnotację @ResponseBody, to dane odsyłane są w odpowiedzi w formacie JSON.
        return LocalTime.now();
    }
    
    @GetMapping("/time2")
    @ResponseBody
    public String time2() {
        // Gdy metoda zwraca String i posiada adnotację @ResponseBody, to dane odsyłane są w odpowiedzi jako HTML.
        // (tzn. są oznaczone jako text/html, ale nie są automatycznie formatowane).
        return LocalTime.now().toString();
    }
    
    private static final DateTimeFormatter FORMAT_DATY = DateTimeFormatter
            .ofPattern("EEEE, dd.MM.YYYY, 'godzina' HH:mm:ss", new Locale("pl", "PL"));

    // Aby powiedzieć wprost, jaki jest format (content-type) odpowiedzi zwracanej przez metodę, można użyć parametru produces
    @GetMapping(path="/time3", produces="text/plain")
    @ResponseBody
    public String time3() {
        return LocalDateTime.now().format(FORMAT_DATY);
    }
    
    // Jak wysłać w odpowiedzi HTML?
    // 1. Utworzyć bezpośrednio w kodzie Javy... - słabe
    @GetMapping(path="/time4", produces="text/html")
    @ResponseBody
    public String time4() {
        LocalDateTime dt = LocalDateTime.now();
        return String.format(
                """
					<html><body>
						<h1>Data i czas</h1>
						<p>Teraz jest godzina <strong style='color:purple'>%s</strong></p>
						<p>Dzisiejsza data: <strong style='color:blue'>%s</strong></p>
						<p style='color: green'>%s</p>
					</body></html>""",
                dt.toLocalTime(), dt.toLocalDate(), dt.format(FORMAT_DATY));
    }

    // 2. Tworzymy odpowiedź za pomocą "szablonu" z wykorzystaniem dodatkowej technologii,
    // tutaj jest to Thymeleaf, ale może być coś innego: JSP, Velocity, FreeMarker, Mustache, ...

    // Gdy metoda w kontrolerze w wyniku zwraca String, a nie posiada adnotacji @ResponseBody,
    // to Spring traktuje zwracaną wartość jak nazwę szablonu,
    // w odp. miejscu szuka pliku z szablonem i generuje wynik za pomocą tego szablonu.

    // Gdy do szablonu chcemy przekazać jakieś dane, to dodajemy je jako atrybuty do modelu.
    // Najprościej zrobić to poprzez parametr model.
    
    @GetMapping("/time5")
    public String time5(Model model) {
    	LocalDateTime dataCzas = LocalDateTime.now();
    	model.addAttribute("dt", dataCzas);
    	return "wyswietl_czas5.html";
    }
    
    // Przykład pokazujący dalsze możliwości Thymeleaf.
    // https://www.thymeleaf.org/
    @GetMapping("/time6")
    public String time6(Model model) {
        LocalDateTime dt = LocalDateTime.now();
        model.addAttribute("dt", dt);
        
        List<LocalDate> dates = new ArrayList<>();
        LocalDate date = dt.toLocalDate();
        for(int i = 0; i <= 5; i++) {
            dates.add(date.plusWeeks(i));
        }
        model.addAttribute("futureWeeks", dates);

        return "wyswietl_czas6.html";
    }

}

