package sklep.rest;

import java.math.BigDecimal;
import java.util.List;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import sklep.db.DBConnection;
import sklep.db.DBException;
import sklep.db.ProductDAO;
import sklep.db.RecordNotFound;
import sklep.model.Product;

// Adnotacje @Produces i @Consumes można umieścić na poziomie klasy.
// Wszystkie metody, które coś produkują (czyli nie są void), produkują JSON-a
// (wyjątkiem jest photo - na poziomie metody można to ustawienie zmienić)
// Wszystkie metody, które coś konsumują (czyli posiadają parametr, w którym odbierane jest "ciało zapytania") konsumują JSON.
@Path("/products.json")
@Produces("application/json")
@Consumes("application/json")
public class RProductsJSON {
    
    @GET
    public List<Product> readAll() throws DBException {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            return productDAO.readAll();
        }
    }
    
    @GET
    @Path("/{id}")
    public Product readOne(@PathParam("id") int productId) throws DBException, RecordNotFound {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            return productDAO.findById(productId);
        }
    }
    
    @GET
    @Path("/{id}/price")
    public BigDecimal getPrice(@PathParam("id") int productId) throws DBException, RecordNotFound {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            Product product = productDAO.findById(productId);
            return product.getPrice();
        }
    }
    
    @PUT
    @Path("/{id}/price")
    public void putPrice(@PathParam("id") int productId,
            BigDecimal newPrice) throws DBException, RecordNotFound {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            Product product = productDAO.findById(productId);
            product.setPrice(newPrice);
            productDAO.update(product);
            db.commit();
        }
    }
    
    // (w tej klasie) - odesłanie uzupełnionego rekordu - trochę niewydajne, ale wygodne
    @POST
    public Product zapiszProdukt(Product product) throws DBException {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            productDAO.save(product);
            db.commit();
            return product;
        }
    }
    
    @DELETE
    @Path("/{id}")
    public void usun(@PathParam("id") int productId) throws DBException, RecordNotFound {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            productDAO.delete(productId);
            db.commit();
        }
    }
    
    @GET
    @Produces("image/jpeg")
    @Path("/{id}/photo")
    public byte[] photo(@PathParam("id") int productId) throws DBException, RecordNotFound {
        return PhotoUtil.readBytes(productId);
    }
    
}
