package sklep.rest;

import java.math.BigDecimal;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.server.ResponseStatusException;

import sklep.model.Product;
import sklep.repository.ProductRepository;
import sklep.util.PhotoUtil;

@RestController
@RequestMapping("/rest/products")
public class ProductEndpoint {
    @Autowired
    private ProductRepository productRepository;
    
    @Autowired
    private PhotoUtil photoUtil;
    
    @GetMapping
    public List<Product> readAll() {
        return productRepository.findAll(Sort.by("productId"));
    }
    
    @GetMapping("/{id}")
    public Product readOne(@PathVariable Integer id) {
        return productRepository.findById(id).orElseThrow(() -> new ResponseStatusException(HttpStatus.NOT_FOUND));
    }
    
    // Ta metoda pokazana tylko po to, aby wytłumaczyć, że struktura adresów powinna odpowiadać logicznej strukturze danych.
    // W prawdziwej aplikacji raczej nie dochodzi się do poziomu pojedynczych pól. Teoretycznie można.
    @GetMapping("/{id}/price")
    public BigDecimal getPrice(@PathVariable Integer id) {
        return readOne(id).getPrice();
    }
    
    /*
     * Operacja PUT służy do zapisania danych POD PODANYM ADRESEM.
     * Na przykład PUT products/2/price z wartością 100 powinno ustawić w produkcie nr 2 cenę 100.
     * Jeśli PUT zadziała, to następnie GET wysłany pod ten sam adres powinien odczytać te same dane,
     * które PUT zapisał (być może w innym formacie - to inny temat)
     * 
     * PUT najczęściej jest używany do aktualizacji istniejących danych
     * (pojedynczych wartości albo całych rekordów), ale może być też użyty do
     * zapisania nowych danych. To, co najważniejsze, to fakt, że PUT zapisuje dane
     * pod konkretnym adresem, do którego jest wysyłany.
     * 
     * Aby odczytać dane, które przysłał klient, metoda ma jeden parametr oznaczony @RequestBody.
     * To do tego parametru Spring przekaże dane odczytane z "body" zapytania.
     */
    @PutMapping("/{id}/price")
    public void setPrice(@PathVariable Integer id, @RequestBody BigDecimal newPrice) {
        Product product = productRepository.findById(id).orElseThrow(() -> new ResponseStatusException(HttpStatus.NOT_FOUND));
        product.setPrice(newPrice);
        productRepository.save(product);
    }
    
    /*
     * POST jest najbardziej ogólną metodą HTTP; oznacza, że klient
     * "wysyła jakieś dane na serwer", a serwer odsyła jakąś odpowiedź.
     * W zasadzie POST może służyć do wszystkiego.
     * 
     * W praktyce POST bardzo często służy do dodawania nowych rekordów, ponieważ
     * gdy tworzymy nowy rekord, to nie znamy z góry jego ID i nie wiemy pod jakim
     * URL-em zostanie zapisany (nie da się więc użyć PUT). Stąd wzięła REST-owa
     * konwencja, że aby dodać nowy rekord do katalogu, wysyła się POST z danymi
     * tego rekordu pod ogólny adres całego katalogu.
     */
    @PostMapping
    public Product insert(@RequestBody Product product) {
    /* Operacja save (a wewnętrznie persist z Hibernate) spowoduje ustawienie nowego ID w obiekcie.
     * Warto taką informację przekazać klientowi. Można:
     * 1) odesłać uzupełniony rekord (i tak zrobimy tutaj),
     * 2) odesłać "małego JSON-a" z informacją o tym ID
     *    (i innymi informacjami, które serwer chce przekazać klientowi)
     * 3) tylko nagłówek Location z URL-em nowego rekordu (to zobaczymy w wersji JAX-RS).
     */
        productRepository.save(product);
        return product;
    }

    @DeleteMapping("/{id}")
    public void delete(@PathVariable("id") Integer productId) {
        try {
            productRepository.deleteById(productId);
        } catch (EmptyResultDataAccessException e) {
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, "Brak produktu nr " + productId);
        }
    }
    
    /* Większość komunikacji w usługach REST odbywa się w formacie JSON,
     * ale czasami używany jest też format XML,
     * a dla niektórych danych stosujemy bezpośrednio jakiś format specjalny, np. PNG, JPG dla obrazów, PDF dla wydruków itp.
     */
    @GetMapping(path="/{id}/photo", produces="image/jpeg")
    public byte[] getPhoto(@PathVariable Integer id) {
        return photoUtil.readBytes(id);
    }

}
