package sklep.controller;

import jakarta.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.*;
import sklep.basket.Basket;
import sklep.model.Product;
import sklep.repository.ProductRepository;
import sklep.util.PhotoUtil;

import java.math.BigDecimal;
import java.util.List;
import java.util.Optional;


@Controller
@RequestMapping("/products")
public class ProductController {
    @Autowired
    private ProductRepository productRepository;

    @Autowired
    private PhotoUtil photoUtil;

    @GetMapping
    public String readAll(Model model) {
        List<Product> products = productRepository.findAll();
        model.addAttribute("products", products);
        return "products";
    }

    @GetMapping("/{id}")
    public String readOne(@PathVariable("id") Integer productId, Model model) {
        Optional<Product> product = productRepository.findById(productId);
        if(product.isPresent()) {
            model.addAttribute("product", product.get());
            return "product";
        } else {
            return "missing_product";
        }
    }

    @GetMapping("/{id}/edit")
    public String editProduct(@PathVariable("id") Integer productId, Model model) {
        Optional<Product> product = productRepository.findById(productId);
        if(product.isPresent()) {
            model.addAttribute("product", product.get());
            return "product_form";
        } else {
            return "missing_product";
        }
    }

    @GetMapping("/new")
    public String newProduct(Product product) {
        return "product_form";
    }

    @PostMapping({"/{id}/edit", "/new"})
    public String saveProduct(@Valid Product product,
                              BindingResult bindingResult) {
        // W tej wersji dane z wypełnionego formularza odbieramy w postaci jednego obiektu Product.
        // Na taki obiekt mówi się czasami "form backing bean".
        // Spring sam wpisze dane do pól o takich samych nazwach.
        // Taki parametr od razu staje się częścią modelu (to jest tzw. ModelAttribute)

        // Kwestia Bean Validation:
        // W tej wersji przed parametrem Product jest adnotacja @Valid
        // i dodatkowo do metoda posiada parametr BindingResult.
        // Wówczas Spring dokonuje walidacji obiektu przed wywołaniem tej metody
        // i informacje o wyniku walidacji przekazuje w parametrze BindingResult.
        // Metoda jest wywoływana zawsze, a to programista ma sprawdzić czy walidacja się powiodła.
        // W BindingResult znajdują się też informacje o błędach.
        if(bindingResult.hasErrors()) {
            System.err.println("Błędy w formularzu: " + bindingResult.getAllErrors());
            // nie próbujemy robić save, pozostajemy w formularzu, w którym powinny się wyświetlić błędy
            return "product_form";
        } else {
            // Gdy próbujemy wywołać save, a obiekt nie spełnia wymagań validation, to wtedy Hibernate zablokuje taki zapis (wyrzuci wyjątek).
            // W tej wersji do tego nie dojdzie, bo if sprawdził, że nie ma błędów walidacji. Mogą być błedy techniczne bazy danych.
            productRepository.save(product);
            // Po pomyślnym zapisaniu przechodzimy na stronę tego produktu
            return "redirect:/products/" + product.getProductId();
        }
    }

    @GetMapping("/{id}/add-to-basket")
    public String addToBasket(
            @PathVariable("id") int productId,
            @SessionAttribute Basket basket){
        Optional<Product> product = productRepository.findById(productId);
        if(product.isPresent()) {
            basket.addProduct(product.get());
        }
        return "redirect:/products";
    }

    @GetMapping("/szukaj")
    public String szukaj(Model model,
                                  String name,
                                  BigDecimal min,
                                  BigDecimal max) {
        List<Product> products = List.of();
        if(name != null && !name.isEmpty() && min == null && max == null) {
            products = productRepository.findByProductNameContainingIgnoreCase(name);
        } else if ((name == null || name.isEmpty()) && (min != null || max != null)) {
            if(min == null) {
                min = BigDecimal.ZERO;
            }
            if(max == null) {
                max = BigDecimal.valueOf(1000_000_000);
            }
            products = productRepository.findByPriceBetween(min, max);
        } else if (name != null && !name.isEmpty() && (min != null || max != null)) {
            if(min == null) {
                min = BigDecimal.ZERO;
            }
            if(max == null) {
                max = BigDecimal.valueOf(1000_000_000);
            }
            products = productRepository.findByProductNameContainingIgnoringCaseAndPriceBetween(name, min, max);
        }
        model.addAttribute("products", products);
        return "wyszukiwarka2";
    }

    @GetMapping(path="/{id}/photo", produces="image/jpeg")
    @ResponseBody
    public byte[] getPhoto(@PathVariable("id") Integer productId) {
        return photoUtil.readBytes(productId);
    }
}
