package sprzedaz.wydajnosc;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.nio.file.Files;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/* Naważniejsza rzecz - w tej wersji korzystamy ze String.intern()
 Każdy wczytywany z pliku napis zamieniamy na napis o tej samej treści "z puli".
 JVM utrzymuje pulę Stringów używanych przez aplikację.
 Dla każdego napisu można odnaleźć odpowiadający mu napis z puli - wszystkie wystąpienia tego samego napisu
 (np. "Warszawa") zostaną zastąpione odwołaniami do TEGO SAMEGO obiektu w pamięci.
 Dzięki temu po wczytaniu dane zajmują mniej pamięci, GC ma mniej roboty, szybciej działa porównywanie napisów (a co za tym idzie np. HashMapa).
 */ 
public class WczytajSprzedaz5 {

	static List<Transakcja> wczytaj(File plik) throws IOException {
		Map<String, LocalDate> pulaDat = new HashMap<>();
		Map<String, BigDecimal> pulaCen = new HashMap<>();

		List<Transakcja> lista = new ArrayList<>();
		try(BufferedReader reader = Files.newBufferedReader(plik.toPath())) {
			String linia;
			while ((linia = reader.readLine()) != null) {
				String[] pola = linia.split(",");
				int sztuk = Integer.parseInt(pola[6]);
				
				LocalDate data = pulaDat.get(pola[0]);
				if(data == null) {
					data = LocalDate.parse(pola[0]);
					pulaDat.put(pola[0], data);
				}
				
				BigDecimal cena = pulaCen.get(pola[5]);
				if(cena == null) {
					cena = new BigDecimal(pola[5]);
					pulaCen.put(pola[5], cena);
				}
				
				String miasto = pola[1].intern();
				String sklep = pola[2].intern();
				String kategoria = pola[3].intern();
				String towar = pola[4].intern();
				
				Transakcja transakcja = new Transakcja(data, miasto, sklep, kategoria, towar, cena, sztuk);
				lista.add(transakcja);
			}
		}
		return lista;
	}

	public static void main(String[] args) {
		Pomiary.wypiszPamiec();

		Opakowanie<List<Transakcja>> wynik = Opakowanie.empty();

		System.out.println("\nczytam");
		File plik = new File("sprzedaz_100.csv");
		
		Pomiary.zmierzCzasDzalania(() -> {
			try {
				List<Transakcja> lista = wczytaj(plik);
				wynik.set(lista); // aby GC nie usunął listy
				System.out.println("Wczytano " + lista.size());
			} catch (IOException e) {
				e.printStackTrace();
			}
		});
		Pomiary.wypiszPamiec();

		System.out.println("\ngc");
		System.gc();
		try {
			Thread.sleep(2000);
		} catch (InterruptedException e) {
		}
		Pomiary.wypiszPamiec();
		System.out.println(wynik.get().size());
	}

}
