--* Funkcje agregujące *--

-- W zapytaniach można używać funkcji
SELECT lower(first_name), upper(last_name), sqrt(salary)
FROM employees;

-- ALE niektóre funkcje to są "funkcje agregujące" i one wpływają na zapytanie w taki sposób,
-- że zamiast wielu oddzielnych wyników, widzimy jeden wynik zbiorczy:
SELECT avg(salary)
FROM employees;

-- Zauważmy, że tylko sama nazwa funkcji decyduje o tym zachowaniu. Inna funkcja liczbowa daje 107 oddzielnych wyników.
-- Na takie funkcje mówimy "funkcja skalarna".
SELECT sqrt(salary)
FROM employees;

-- Istnieje 5 kanonicznych funkcji agregujących - dostępne w każdej bazie SQL:
SELECT count(salary), sum(salary), min(salary), avg(salary), max(salary)
FROM employees;

-- Funkcji count można używać na 3 sposoby:
-- count(*) - zwraca liczbę rekordów
-- count(kolumna) - ile niepustych (nie-NULLowych) wartości jest w tej kolumnie
-- count(distinct kolumna) - ile jest różnych niepustych wartości jest w tej kolumnie
SELECT count(*), count(department_id), count(DISTINCT department_id)
FROM employees;

/* Lista funkcji agregujących PostgreSQL:
   https://www.postgresql.org/docs/current/functions-aggregate.html
 */
SELECT avg(salary), stddev(salary), variance(salary) FROM employees;

SELECT every(length(street_address) > length(city)) FROM locations;

SELECT * FROM locations
WHERE NOT length(street_address) > length(city);

SELECT string_agg(city, '; ') FROM locations;

SELECT string_agg(city, '; ') FROM (SELECT city FROM locations ORDER BY city) podzapytanie;
SELECT string_agg(DISTINCT last_name, ', ') FROM employees;

SELECT json_agg(city) FROM locations;

-- PostgreSQL pozwala na używanie słowa DISTINCT w większości funkcji agregujących.
-- Wtedy powstarzące się wartości są liczone tylko raz (jako jedna wartość).
-- Przeciwieństwem DISTINCT jest ALL - "licz wszystkie wartości", ale to jest ustawienie domyślne i tego się nie pisze.
SELECT avg(salary), avg(ALL salary), avg(DISTINCT salary)
FROM employees;

-- W PostgreSQL można też za wywołaniem funkcji agregującej dopisać FILTER
-- i warunek ograniczający zbiór wartości uwzględnianych przez funkcję.
SELECT count(*) AS wszyscy
    , count(*) FILTER (WHERE salary >= 10000) AS bogaci
    , count(*) FILTER (WHERE salary < 10000) AS biedni
FROM employees;

-- BTW, to da się zrobić jako podzapytanie i wtedy zadziała także w innych bazach danych.
SELECT (SELECT count(*) FROM employees) AS wszyscy
, (SELECT count(*) FROM employees WHERE salary >= 10000) AS bogaci
, (SELECT count(*) FROM employees WHERE salary < 10000) AS biedni
-- Oracle: FROM dual
;

-- Jeśli f.agr. połączymy z filtrowaniem WHERE, możemy obliczyć statystyki dla wybranych rekordów.
-- W szczególności można poznać liczbę rekordów spełniających warunek.

-- min, max, i średnia pensja programistów:
SELECT count(*), min(salary), max(salary), avg(salary)
FROM employees
WHERE job_id = 'IT_PROG';

--* Grupowanie *--

-- GROUP BY dzieli dane na grupy zwn na wartość podanego kryterium i dla każdej grupy oblicza statystyki / funkcje agregujące
-- W SELECT można wypisać wartość, po której grupowaliśmy, a do pozostałych kolumn trzeba zastosować funkcję agregującą.
SELECT job_id, count(*), min(salary), max(salary), avg(salary)
FROM employees
GROUP BY job_id;

-- porównajmy:
-- wartość bez agregacji - 107 oddzielnych wartości
SELECT salary FROM employees;

-- agregacja całej tabeli - pojedyncza wartość wynikowa
SELECT avg(salary) FROM employees;

-- grupowanie - tyle wyników, ile różnych wartości wybranego kryterium wystepuje w danych
SELECT avg(salary) FROM employees GROUP BY job_id;


-- HAVING to jest warunek nakładany na grupy, a nie na pojedyncze rekordy
-- spośród grup opartych o job_id wypisz tylko co najmniej 10-osobowe
SELECT job_id, count(*), min(salary), max(salary), avg(salary)
FROM employees
GROUP BY job_id
HAVING count(*) >= 10;

-- WHERE - warunki dot. pojedynczych rekordów stosowane przed grupowaniem
-- HAVING - warunki dotyczące całych rekordów, po grupowaniu

-- Bierzemy pracowników, którzy zarabiają < 6 tys, grupujemy wg stanowisk
-- i następnie wyliczamy średnią tylko dla tych pracowników.
-- Zauważmy, że:
-- 1) wypisuje się stanowisko ST_MAN, bo jeden z ST_MANów zarabia < 6 tys.
-- 2) dla stanowiska IT_PROG średnia wynosi 4600, bo uwzględniamy tylko tych 3 programistów, którzy zarabiają < 6tys.
SELECT job_id, count(*) AS ilu, avg(salary) AS srednia, min(salary) AS min, max(salary) AS max
FROM employees
WHERE salary < 6000
GROUP BY job_id;

-- Wszystkich pracowników grupujemy wg stanowisk i wyliczamy średnią dla całych grup.
-- A następnie wyświetlamy tylko te grupy (te stanowiska), na których średnia pensja > 6 tys.
-- Zauważmy, ze:
-- 1) NIE wypisuje się stanowisko ST_MAN, bo średnia ST_MANów wynosi 7280 i nie przechodzi przez HAVING
-- 2) dla stanowiska IT_PROG średnia wynosi 5760, bo to średnia wszystkich programistów
SELECT job_id, count(*) AS ilu, avg(salary) AS srednia, min(salary) AS min, max(salary) AS max
FROM employees
GROUP BY job_id
HAVING avg(salary) < 6000;

-- Przy okazji: Zamiast HAVING można też umieści zapytanie z GROUP BY jako podzapytanie
-- w zewnętrznym zapytaniu, a w tym zewnętrznym użyć WHERE.
SELECT * FROM (
    SELECT job_id, count(*) AS ilu, avg(salary) AS srednia, min(salary) AS min, max(salary) AS max
    FROM employees
    GROUP BY job_id) podzapytanie
WHERE srednia < 6000;

-- NULL w grupowaniu tworzy oddzielną grupę
SELECT department_id, count(*), min(salary), max(salary), avg(salary)
FROM employees
GROUP BY department_id;

-- Zasadniczo gdy użyjemy funkcji agregującej lub zrobimy GROUP BY, to w klauzuli SELECT (oraz HAVING i ORDER BY)
-- możemy odwoływać się już tylko do:
-- - bezpośrednio wartości, po których grupowaliśmy
-- - pozostałych kolumn tylko poprzez funkcje agregujące
-- złe:
SELECT first_name, last_name, avg(salary)
FROM employees;

-- też złe:
SELECT first_name, last_name, job_id, max(salary)
FROM employees
GROUP BY job_id;

-- w SELECT wolno używać tylko to, po czym grupowaliśmy
-- (lub to, co jednoznacznie wynika z kolumn, po których grupowaliśmy - jeszcze będzie o tym mowa)

-- PostgreSQL pozwala uzywać aliasów kolumn w GROUP BY (Oracle nie pozwala)
SELECT extract(YEAR FROM hire_date) AS rok, count(*) AS ilu
FROM employees
GROUP BY rok
ORDER BY rok;

-- Zadania, które będą miały "ciąg dalszy..."
-- Stosujemy grupowanie wraz z łączeniem tabel...
-- Takie połączenie tabel pozwala uzyskać dostęp do kolumny job_title:
SELECT * FROM employees JOIN jobs USING(job_id);

-- Napisz zapytanie, które dla każdego stanowiska oblicza statystki pracowników:
-- count, min(salary), max(salary), avg(salary)
-- Należy wypisać job_title i te statystyki...
SELECT job_title
    , count(*) AS ilu
    , min(salary) AS minimalna
    , max(salary) AS maksymalna
    , avg(salary) AS srednia
FROM employees JOIN jobs USING(job_id)
GROUP BY job_title;

-- numer kolumny, po której grupujemy
SELECT job_title
    , count(*) AS ilu
    , min(salary) AS minimalna
    , max(salary) AS maksymalna
    , avg(salary) AS srednia
FROM employees JOIN jobs USING(job_id)
GROUP BY 1;

-- Zasadniczo w SELECT można używać tylko tych kolumn, po ktrych się grupowało
-- (a pozostałych tylko porpzez f. agr.). Np. Oracle przestrzega tego bardzo ściśle.
SELECT job_id, job_title
    , count(*) AS ilu
    , min(salary) AS minimalna
    , max(salary) AS maksymalna
    , avg(salary) AS srednia
FROM employees JOIN jobs USING(job_id)
GROUP BY job_id, job_title;

-- Pod pewnymi warunkami PostgreSQL jednak pozwala odwołać się do wartości,
-- która jednoznacznie wynika z tego, po czym grupowaliśmy.
SELECT job_id, job_title
    , count(*) AS ilu
    , min(salary) AS minimalna
    , max(salary) AS maksymalna
    , avg(salary) AS srednia
FROM jobs JOIN employees USING(job_id)
GROUP BY job_id;

-- Dalszy ciąg zadania...
-- Analogiczne zapytanie, ale dla departamentów.
-- Chcemy, aby wypisały się nazwy wszystkich departamentów, także tych, w których nikt nie pracuje (np. Payroll).
-- Chcemy mieć informację o liczbie pracowników tego departamentu oraz średniej pensji (min max też można, ale to nie jest najważniejsze).
-- Takie zapytanie dałoby bezsensowne wyniki - jedynka dla departamentów, w których nikt nie pracuje
SELECT department_id, department_name
    , count(*) AS ilu
    , min(salary) AS minimalna
    , max(salary) AS maksymalna
    , avg(salary) AS srednia
FROM departments LEFT JOIN employees USING(department_id)
GROUP BY department_id
ORDER BY department_id;

-- Zamiast count(*) trzeba użyć count(employee_id) lub count(employees)
-- A najlepiej napisać count(DISTINCT employee_id)
SELECT department_id, department_name
    , count(DISTINCT employee_id) AS ilu
    , min(salary) AS minimalna
    , max(salary) AS maksymalna
    , avg(salary) AS srednia
FROM departments LEFT JOIN employees USING(department_id)
GROUP BY department_id
ORDER BY department_id;


--* Grupowanie po wielu kryteriach, CUBE, ROLLUP, GROUPING SETS *--
-- <początek przykładów dot. tabeli sprzedaz>
SELECT * FROM sprzedaz;

-- specyfiką tej tabeli jest to, że istnieje mało wartości unikalnych, a za to one wiele razy się powtarzają
SELECT count(DISTINCT miasto), count(DISTINCT sklep), count(DISTINCT kategoria), count(DISTINCT towar)
FROM sprzedaz;

-- wartością transakcji jest iloczyn cena * sztuk
-- wyświetlmy taką kolumnę obok kolumn odczytanych z tabeli
-- btw: W Oracle nie wolno dopisywać kolumn po przecinku za *
-- SELECT *, cena * sztuk as wartosc FROM sprzedaz;

-- ale można, jeśli użyje się nazwy lub aliasu tabeli:
SELECT sprzedaz.*, cena*sztuk as wartosc FROM sprzedaz;

-- Oblicz sumę wartości wszystkich transakcji z całego pliku
SELECT sum(cena * sztuk) AS suma FROM sprzedaz;

-- Grupowanie zwn pojedyncze kryterium:
SELECT miasto, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY miasto;

SELECT towar, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY towar;

-- Można też grupować zwn na kilka kryteriów jednocześnie
-- Gdy podamy po prostu dwa kryteria po przecinku, to wyliczane statystyki dla każdej pary wartości,
-- jaka jest możliwa (o ile występuje w danych)
-- 72 wiersze
SELECT miasto, towar, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY miasto, towar
ORDER BY miasto, towar;

-- 223 wierszy
-- (w sklepie Podsiało nie ma mebli... ;) )
SELECT miasto, sklep, kategoria, towar,
    count(*) AS l_transakcji, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY miasto, sklep, kategoria, towar
ORDER BY miasto, sklep, kategoria, towar;


-- konstrukcje CUBE, ROLLUP oraz GROUPING SET pozwalają dodać do wyników dodatkowe wiersze, "podsumy"
-- inaczej mówiąc, w jednym zapytaniu wykonujemy grupowanie na wielu poziomach

-- CUBE generuje wszystkie możliwe kombinacje kryteriów.
-- W tym przypadku oprócz wyników normalnego grupowania, mamy tez:
-- 1) dla każdego miasta podsumowanie tego miasta bez względu na towar ("wszystkie towary razem wzięte w danym mieście")
-- 2) dla każdego towaru sumę tego towaru bez względu na miasto
-- 3) sumę wszystkich rekordów z całej tabeli
-- W miejscu tego kryterium, które jest aktualnie pomijane, wstawiany jest NULL
SELECT miasto, towar
    , count(*) AS ile
    , sum(cena * sztuk) AS wartosc
FROM sprzedaz
GROUP BY CUBE(miasto, towar)
ORDER BY miasto, towar;

-- Gdy kryteriów grupowania jest dużo, to CUBE powoduje wygenerowanie bardzo wielu dodatkowych wierszy
-- bo na każdej pozycji może być NULL. Jednak zwykle jest tak, że niektóre częściowe podsumowania nie mają sensu.
-- 1288 wierszy
SELECT miasto, sklep, kategoria, towar,
    count(*) AS l_transakcji, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY CUBE(miasto, sklep, kategoria, towar)
ORDER BY miasto, sklep, kategoria, towar;


-- ROLLUP dodaje wiersze z podsumami, ale tylko idąc od lewej do prawej; kolejność kryteriów grupowania ma tu znaczenie
-- W tym przykładzie będzie podobnie jak w CUBE, ale bez punktu 2 - podsumowań dla towarów
-- 81 wierszy
SELECT miasto, towar
    , count(*) AS ile
    , sum(cena * sztuk) AS wartosc
FROM sprzedaz
GROUP BY ROLLUP(miasto, towar)
ORDER BY miasto, towar;

-- 331 wierszy
SELECT miasto, sklep, kategoria, towar,
    count(*) AS l_transakcji, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY ROLLUP(miasto, sklep, kategoria, towar)
ORDER BY miasto, sklep, kategoria, towar;


-- CUBE opłaca się używać wtedy, gdy kryteria grupowania są od siebie niezależne logicznie,
-- tak jak miast o i towar

-- Nie ma sensu używać CUBE gdy kryteria tworzą logiczna zależność, np. towary należą do różnych kategorii
-- Tutaj: nie ma sensu podsumowanie "wszystkich biurek z dowolnej kategorii", bo i tak każde biurko należy do kategorii meble
SELECT kategoria, towar
    , count(*) AS ile
    , sum(cena * sztuk) AS wartosc
FROM sprzedaz
GROUP BY CUBE(kategoria, towar)
ORDER BY kategoria, towar;

-- Tutaj sens ma ROLLUP
SELECT kategoria, towar
    , count(*) AS ile
    , sum(cena * sztuk) AS wartosc
FROM sprzedaz
GROUP BY ROLLUP(kategoria, towar)
ORDER BY kategoria, towar;

-- W ROLLUP znaczenie ma kolejność kryteriów.
-- Powinniśmy iść od lewej do prawej podając najpierw kryteria "nadrzędne", a później coraz bardziej "podrzędne".
-- To jest bez sensu:
SELECT towar, kategoria, count(*) AS l_transakcji, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY ROLLUP(towar, kategoria)
ORDER BY towar, kategoria;


-- Można też użyć GROUPING SET i samodzielnie określić jakie grupowania mają być przeprowadzone
-- To zapytanie jest równoważne ROLLUP(miasto, towar)
SELECT miasto, towar, count(*) AS l_transakcji, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY GROUPING SETS((miasto, towar), (miasto), ())
ORDER BY miasto, towar;

-- GROUPING SETS daje efekt, jak byśmy wykonali wiele zapytań z GROUP BY takich, jak podany w nawiasach,
-- uzupełni brakujące kolumny NULLAMI
-- i wszystko połączyli za pomocą UNION ALL
-- To samo, co wyżej:
SELECT miasto, towar, count(*) AS l_transakcji, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY miasto, towar
UNION ALL
SELECT miasto, NULL, count(*), sum(cena * sztuk)
FROM sprzedaz
GROUP BY miasto
UNION ALL
SELECT NULL, NULL, count(*), sum(cena * sztuk)
FROM sprzedaz
ORDER BY 1, 2;

-- Wszystkie sensowne grupowania dla czterech kryteriów - więcej niż ROLLUP, ale mniej niż CUBE
-- 439 wierszy
SELECT miasto, sklep, kategoria, towar,
    count(*) AS l_transakcji, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY GROUPING SETS (
    (miasto, sklep, kategoria, towar),
    (miasto, sklep, kategoria),
    (miasto, sklep),
    (miasto, kategoria, towar),
    (miasto, kategoria),
    (miasto),
    (kategoria, towar),
    (kategoria),
    ())
ORDER BY miasto, sklep, kategoria, towar;

-- </koniec przykładów dot. tabeli sprzedaz>

-- Wracamy do employees i na ich przykładzie zajmiemy się tematem rozpoznawiania,
-- który wiersz pochodzi z normalnego grupowania,
-- a który z dodatkowej podsumy...

-- Problem - ponieważ Kimberely Grant ma wpisany NULL w department_id,
-- to wartość NULL w wyniku CUBE lub ROLLUP będzie pojawiać się dwa razy.
-- Raz jako "prawdziwe department_id", a drugi raz jako "wygenerowany dodatkowy wiersz, w którym department_id nie ma znaczenia".
SELECT department_id, job_id, count(*), avg(salary)
FROM employees
GROUP BY CUBE(department_id, job_id)
ORDER BY department_id, job_id;


-- Jak odróżnić sytuację gdzie null faktycznie występował w danych (np. department_id dla K.Grant)
-- od sytuacji gdy ten null jest wypełniaczem w wierszu dodatkowego podsumowania ("dla wszystkich departamentów razem wziętych...")
-- Ta funkcja zwraca 0, gdy wartością danej kolumny jest faktyczna wartość pochodząca z tabeli
-- a 1 gdy pomijamy wartości tej kolumny, bo robimy podsumowanie "bez względu na tę kolumnę" (i jest w niej sztucznie dodany null)
SELECT department_id, grouping(department_id) AS grouping_dep
    , job_id, grouping(job_id) AS grouping_job
    , count(*) AS ilu
    , sum(salary) AS suma
FROM employees
GROUP BY CUBE(department_id, job_id)
ORDER BY grouping_dep, department_id, grouping_job, job_id;

-- Połącz odpowiednie tabele i pogrupuj po 3 kryteriach używając ROLLUP:
-- city, department_name, job_title
-- W tej wersji null pojawia się zarówno z powodu podsumowania, jak i gdy występuje w danych
SELECT city, department_name, job_title
    , count(*) AS ilu
    , sum(salary) AS suma
    , round(avg(salary), 2) AS srednia
FROM employees
    LEFT JOIN departments USING(department_id)
    LEFT JOIN locations USING(location_id)
    LEFT JOIN jobs USING(job_id)
GROUP BY ROLLUP(city, department_name, job_title)
ORDER BY city, department_name, job_title;

-- Za pomocą CASE i funkcji GROUPING możemy wpisać specjalny tekst w miejscach podsumowań
SELECT CASE WHEN grouping(l.city) = 1 THEN '--MIASTA RAZEM--' ELSE l.city END AS city
    , CASE WHEN grouping(d.department_name) = 1 THEN '--DEPARTAMENTY RAZEM--' ELSE d.department_name END AS department_name
    , CASE WHEN grouping(j.job_title) = 1 THEN '--JOBY RAZEM--' ELSE j.job_title END AS job_title
    , count(*) AS ilu
    , sum(salary) AS suma
    , round(avg(salary), 2) AS srednia
FROM employees e
    LEFT JOIN departments d USING(department_id)
    LEFT JOIN locations l USING(location_id)
    LEFT JOIN jobs j USING(job_id)
GROUP BY ROLLUP(l.city, d.department_name, j.job_title)
ORDER BY l.city, d.department_name, j.job_title;

--* Temat Tabel przestawnych" *--
-- Czyli tworzenia kolumn na podstawie wierszy w celu dwuwymiarowanej przezentacji danych.
-- Takie zapytanie zwraca dane w wymiarze pionowym
SELECT miasto, kategoria, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY miasto, kategoria
ORDER BY miasto, kategoria;

-- w PostgreSQL można skorzystać z funkcji crosstab,
-- która należy do rozszerzenia table_func. W praktyce wystarczy je aktywować.
-- (tylko raz dla danej bazy danych, a nie trzeba w każdej nowej sesji)
CREATE EXTENSION IF NOT EXISTS tablefunc;
-- https://www.postgresql.org/docs/current/tablefunc.html

SELECT * FROM crosstab('
SELECT miasto, kategoria, sum(cena * sztuk) AS suma
FROM sprzedaz
GROUP BY miasto, kategoria
ORDER BY miasto, kategoria')
AS sumy(miasto VARCHAR(100),
        meble NUMERIC,
        "szkolno-biurowe" NUMERIC,
        "wyposażenie szkolne" NUMERIC);


-- Informacyjnie: W Oraclu można zrobić to tak:
SELECT * FROM (
    SELECT miasto, kategoria, round(cena * sztuk) AS wartosc
    FROM sprzedaz
) PIVOT (sum(wartosc)
  FOR kategoria IN (
    'meble' AS "meble",
    'szkolno-biurowe' AS "szkb",
    'wyposażenie szkolne' AS "wypszk"
  )
);

-- Wersja Python / Pandas
-- sprzedaz.pivot_table(columns=['kategoria', 'towar']
--         index=['miasto', 'sklep'], values='wartosc', aggfunc='sum', margins=True)
