 --* Operacje na zbiorach ("operacje teoriomnogościowe") *--

-- Przykład: nazwy krajów i miast
SELECT DISTINCT city FROM locations; -- 23
SELECT DISTINCT country_name FROM countries; -- 25

-- Jest jedna nazwa, która występuje w obu tabelach - Singapore

-- UNION ALL - konkatenacja rekordów
-- do wyników jednego zapytania dołączane są pod spodem wyniki kolejnego zapytania

-- wynik obejmuje 48 wierszy - Singapore jest 2 razy
SELECT city FROM locations
UNION ALL
SELECT country_name FROM countries;

-- UNION - suma zbiorów
-- w odróżnieniu od UNION ALL, wynik UNION nie zawiera powtórzeń
-- 47 wartości - Singapore tylko 1 raz
SELECT city FROM locations
UNION
SELECT country_name FROM countries;

-- INTERSECT - iloczyn zbiorów, przecięcie, część wspólna
-- wynikiem jest Singapore
SELECT city FROM locations
INTERSECT
SELECT country_name FROM countries;

-- EXCEPT - różnica zbiorów
-- 22 wartości - wszystkie miasta bez Singapuru
SELECT city FROM locations
EXCEPT
SELECT country_name FROM countries;

-- 24 wartości - wszystkie kraje bez Singapuru
SELECT country_name FROM countries
EXCEPT
SELECT city FROM locations;

-- Zgodnie ze standardem SQL powinno być EXCEPT, tak jest m.in. w PostgreSQL,
-- w starszych wersjach Oracle działał tylko "MINUS",
-- ale w Oracle 21 działa też już słowo EXCEPT


-- Gdy używa się operatorów zbiorów, to:
-- * nazwy kolumn sa ustalane w pierwszym SELECT
-- * liczba oraz typ kolumn muszą się zgadzać
-- * SELECTY moga skladac sie z klauzul FROM, WHERE, GROUP BY, HAVING
-- * natomiast ORDER BY moze wystapic tylko na samym koncu i posortuje wszystkie czesci lacznie

-- Najpierw zobaczymy bogatych, potem biednych, ale kazda z tych czesci nie bedzie w zaden specjalny sposob sortowana
SELECT first_name, last_name, salary
FROM employees
WHERE salary >= 10000
UNION ALL
SELECT first_name, upper(last_name), salary
FROM employees
WHERE salary < 5000;

-- Powiedzmy, że chciałbym mieć najpierw bogatych, potem biednych, ale każdą część alfabetycznie,
-- to nie da się w ten sposób, bo nie wolno wpisac ORDER BY przed UNION ALL:
/* SELECT first_name, last_name, salary
FROM employees
WHERE salary >= 10000
ORDER BY last_name
UNION ALL
SELECT first_name, last_name, salary
FROM employees
WHERE salary < 5000
ORDER BY last_name; */

-- Natomiast ORDER BY na końcu sortuje całe wyniki łącznie
SELECT first_name, last_name, salary
  FROM employees
  WHERE salary >= 10000
UNION ALL
  SELECT first_name, last_name, salary
  FROM employees
  WHERE salary < 5000
ORDER BY last_name;

-- A jak zachować informacje z której części zapytania pochodzi dany rekord
SELECT first_name, last_name, salary, 'bogaty' AS status_majatkowy
    FROM employees
    WHERE salary >= 10000
UNION ALL
  SELECT first_name, last_name, salary, 'biedny'
    FROM employees
    WHERE salary < 5000
ORDER BY status_majatkowy DESC, last_name, first_name;

-- Jeśli nie potrzebujemy wyświetlać żadnego fajnego tekstu, tylko chodzi nam o kolejność, w dodatkowej kolumnie może być numer

SELECT first_name, last_name, salary, 1
FROM employees
WHERE salary >= 10000
UNION ALL
SELECT first_name, last_name, salary, 2
FROM employees
WHERE salary < 5000
ORDER BY 4, 2, 1;

-- A ten numer możemy ostatecznie ukryć stosując podzapytanie.
SELECT first_name || ' ' || last_name AS kto, salary
FROM (SELECT first_name, last_name, salary, 1 AS status
      FROM employees
      WHERE salary >= 10000
    UNION ALL
      SELECT first_name, last_name, salary, 2
      FROM employees
      WHERE salary < 5000)
ORDER BY status, last_name, first_name;


-- W wyniki można wpleść dodatkowe wiersze nie pochodzące z bazy danych.
SELECT kto, salary
FROM (
      SELECT 'Bogaci:' AS kto, NULL AS salary, 1 AS kolejnosc
    UNION ALL
      SELECT '  ' || first_name || ' ' || last_name, salary, 2
        FROM employees
        WHERE salary > 10000
    UNION ALL
      SELECT 'Biedni:', NULL, 3
    UNION ALL
      SELECT '  ' || first_name || ' ' || last_name, salary, 4
        FROM employees
        WHERE salary < 5000
    ORDER BY 3, 1);

-- Popatrzmy do tabeli job_history...
-- Tam są informacje, że pracownik X wcześniej pracował w departamencie Y na stanowisku Z
SELECT * FROM job_history;

-- Zadanie:
-- Wypisz dane wszystkich pracowników i dla każdego podaj
-- employee_id, department_id, job_id
-- Oprócz danych o bieżącym statusie (wartość department_id, job_id z tabeli employees)
-- wypisz także informacje historyczne z job_history.

-- W pierwsze wersji wyświetl wyłącznie ID.

SELECT * FROM job_history ORDER BY employee_id, start_date;

SELECT employee_id, department_id, job_id, start_date, end_date
FROM job_history
UNION ALL
SELECT employee_id, department_id, job_id, hire_date, NULL
FROM employees
ORDER BY employee_id, end_date;

-- W drugiej wersji dołącz tabele, aby wypisać department_name i job_title,
-- wypisz imię i nazwisko
-- wypisz daty graniczne (w przypadku stanu bieżącego za początek przyjmij hire_date, a jako koniec NULL)

SELECT e.employee_id, e.first_name, e.last_name,
    d.department_id, d.department_name,
    j.job_id, j.job_title,
    h.start_date, h.end_date
FROM job_history h
    JOIN employees e ON e.employee_id = h.employee_id
    JOIN jobs j ON j.job_id = h.job_id
    JOIN departments d ON d.department_id = h.department_id

UNION ALL

SELECT e.employee_id, e.first_name, e.last_name,
    department_id, d.department_name,
    job_id, j.job_title,
    e.hire_date, NULL
FROM employees e
    JOIN jobs j USING(job_id)
    LEFT JOIN departments d USING(department_id)
ORDER BY employee_id, end_date;


--* Konstrukcja CASE *--

-- dla KAŻDEGO pracownika, chcemy info, czy zarabia co najmnie 10 tys, czy mniej
-- W Oracle nie da się wpisać warunku logicznego w kaluzuli SELECT.
-- Wartości logiczne to nie są normalne wartości takie jak liczby, napisy itp.
-- W innych bazach często istnieje typ boolean - w Oracle nie
SELECT first_name, last_name, salary, salary >= 10000
FROM employees;

-- Konstrukcja CASE pozwala sprawdzić warunek logiczny i w zależności od niego zwrócić jeden lub inny wynik
SELECT first_name, last_name, salary,
    CASE WHEN salary >= 10000 THEN 'bogaty' ELSE 'biedny' END
FROM employees;

-- CASE może mieć wiele gałęzi WHEN i może mieć jedną gałąź ELSE
-- Warunki są sprawdzane po kolei i wykonany zostanie pierwszy WHEN, który jest prawdziwy,
-- a ELSE, jeśli żaden WHEN nie był prawdziwy
-- a jeśli nie ma ELSE, to w tej sytuacji wynikiem jest NULL
-- Na końcu konstrukcji CASE trzeba napisać END

-- CASE ma dwa warianty jeśli chodzi o zapis
-- pierwszy wariant - wpisujemy warunki logiczne podobne do warunków WHERE
SELECT first_name, last_name, salary,
    CASE
        WHEN salary >= 15000 THEN 'bardzo bogaty'
        WHEN salary >= 10000 THEN 'bogaty'
        WHEN salary >= 5000 THEN 'średni'
        ELSE 'biedny'
    END AS status
FROM employees;

-- Jeśli nie ma gałęzi ELSE, a rekord nie pasuje do żadnego WHEN,
-- to wynikiem będzie NULL
SELECT first_name, last_name, salary,
    CASE
        WHEN salary >= 15000 THEN 'bardzo bogaty'
        WHEN salary >= 10000 THEN 'bogaty'
    END AS status
FROM employees;

-- warunek wyliczający się do NULL jest traktowany jak fałsz, podobnie jak w WHERE
SELECT first_name, last_name, commission_pct,
    CASE
        WHEN commission_pct >= 0.2 THEN 'wysoka prowizja'
        ELSE 'niska prowizja'
    END AS status
FROM employees;

SELECT first_name, last_name, commission_pct,
    CASE
        WHEN commission_pct >= 0.2 THEN 'wysoka prowizja'
        WHEN commission_pct < 0.2 THEN 'niska prowizja'
        ELSE 'brak prowizji'
    END AS status
FROM employees;


SELECT first_name, last_name, hire_date,
    CASE
    WHEN extract(month FROM hire_date) = 1 THEN 'styczeń'
    WHEN extract(month FROM hire_date) = 2 THEN 'luty'
    WHEN extract(month FROM hire_date) = 3 THEN 'marzec'
    WHEN extract(month FROM hire_date) = 4 THEN 'kwiecień'
    WHEN extract(month FROM hire_date) = 5 THEN 'maj'
    WHEN extract(month FROM hire_date) = 6 THEN 'czerwiec'
    WHEN extract(month FROM hire_date) = 7 THEN 'lipiec'
    WHEN extract(month FROM hire_date) = 8 THEN 'sierpień'
    WHEN extract(month FROM hire_date) = 9 THEN 'wrzesień'
    WHEN extract(month FROM hire_date) = 10 THEN 'październik'
    WHEN extract(month FROM hire_date) = 11 THEN 'listopad'
    WHEN extract(month FROM hire_date) = 12 THEN 'grudzień'
    END AS miesiac
FROM employees;

-- drugi wariant - dopasowywanie wartości
-- Tylko raz podajemy wartość, którą chcemy sprawdzić (tutaj extract), a w WHEN wartości, do których porównujemy
SELECT first_name, last_name, hire_date,
    CASE extract(month FROM hire_date)
    WHEN 1 THEN 'styczeń'
    WHEN 2 THEN 'luty'
    WHEN 3 THEN 'marzec'
    WHEN 4 THEN 'kwiecień'
    WHEN 5 THEN 'maj'
    WHEN 6 THEN 'czerwiec'
    WHEN 7 THEN 'lipiec'
    WHEN 8 THEN 'sierpień'
    WHEN 9 THEN 'wrzesień'
    WHEN 10 THEN 'październik'
    WHEN 11 THEN 'listopad'
    WHEN 12 THEN 'grudzień'
    ELSE 'niewiadomoco'
    END AS miesiac
FROM employees;

-- Oracle (nie działa w Postgresie) posiada też funkcję decode , która działa podobnie jak drugi wariant CASE'a,
-- ale jest bardziej zwięzła (ale też nieprzenośna).
-- decode(wejście, wartość1, wynik1, wartość2, wynik2, ....)
SELECT first_name, last_name, hire_date,
    decode(extract(month FROM hire_date),
        1  , 'styczeń',
        2  , 'luty',
        3  , 'marzec',
        4  , 'kwiecień',
        5  , 'maj',
        6  , 'czerwiec',
        7  , 'lipiec',
        8  , 'sierpień',
        9  , 'wrzesień',
        10 , 'październik',
        11 , 'listopad',
        12 , 'grudzień') AS miesiac
FROM employees;

SELECT first_name, last_name, job_id,
    decode(job_id, 'IT_PROG', 'programista', 'AD_PRES', 'prezes', 'ST_CLERK', 'magazynier', 'ktoś inny') AS kto
FROM employees
ORDER BY employee_id;

/* Na podstawie wysokości pensji ustalamy stawkę podatkową.
Gdy roczne dochody < 50 tys, to stawka wynosi 0.1
między 50 tys a 100 tys, to stawka wynosi 0.2
między 100 a 150 tys, to stawka wynosi 0.3
między 150 a 200 tys, to stawka wynosi 0.4
powyżej 200 tys 0.5

Najpierw wypisz stawkę dla pracownika
Ale później oblicz roczny podatek - w miarę możliwości oblicz zgodnie z zasadami podatku progresywnego,
że wyższy procent jest liczony tylko od kwoty powyżej progu.
*/

SELECT first_name, last_name, salary, 12*salary AS roczne,
    CASE WHEN 12*salary <  50000 THEN 0.1
        WHEN 12*salary < 100000 THEN 0.2
        WHEN 12*salary < 150000 THEN 0.3
        WHEN 12*salary < 200000 THEN 0.4
        ELSE 0.5
    END AS stawka
FROM employees;

-- przy uproszczonym podejściu trzeba by tę stawkę pomnożyć przez roczne zarobki
-- ale wtedy mielibyśmy efekt "przekroczenia progu o złotówkę"
-- Aby nie pisać 12*salary we wszystkich miejscach, za pomocą WITH obliczymy sobie tę wartość wcześniej
WITH emps_roczne AS (
  SELECT employee_id, first_name, last_name, salary, 12*salary AS roczne
  FROM employees)
SELECT first_name, last_name, salary, roczne,
    CASE WHEN roczne <  50000 THEN 0.1 * roczne
        WHEN roczne < 100000 THEN 0.1*50000 + 0.2 * (roczne - 50000)
        WHEN roczne < 150000 THEN 0.1*50000 + 0.2*50000 + 0.3 * (roczne-100000)
        WHEN roczne < 200000 THEN 0.1*50000 + 0.2*50000 + 0.3*50000 + 0.4*(roczne-150000)
        ELSE 0.1*50000 + 0.2*50000 + 0.3*50000 +  + 0.4*50000 + 0.5*(roczne-200000)
    END AS podatek
FROM emps_roczne;

-- już niedostępne SELECT * FROM emps_roczne;
