-- Zgodnie z najlepszymi praktykami dane w bazach rozdziela się do osobnych tabel i wiąże za pomocą "kluczy obcych".
-- Często w zapytaniach chcemy jednak uzyskać rekordy z jednej tabeli wraz z powiązanymi danymi z innych tabel.

-- zajrzyjmy do tabel
SELECT * FROM employees;
SELECT * FROM jobs;
SELECT * FROM departments;
SELECT * FROM locations;
SELECT * FROM countries;
SELECT * FROM regions;

-- Dla odmiany widok emp_details_view prezentuje dane połączone, w wersji nieznormalizowanej,
-- zauważmy mnóstwo powtórzonych wartości w kolumnach city, country...
SELECT * FROM emp_details_view;

-- Jak samemu odczytać dane rozporoszone po wielu tabelach?
-- We FROM można podać więcej niż jedną tabelę:
SELECT * FROM employees, departments;

-- W takiej sytuacji baza danych zwraca wszystkie możliwe kombinacje rekordów z jednej tabeli z rekordami z drugiej tabeli
-- To jest tzw. "iloczyn kartezjański".
SELECT count(*) FROM employees, departments;
SELECT count(*) FROM employees; -- 107
SELECT count(*) FROM departments; -- 27
SELECT 107 * 27;


-- Aby powiązać pracownika z jego departamentem można:
-- 1) zadać zapytanie do obu tabel (rozdziela przecinek) a następnie w WHERE podać "warunek złączenia"
SELECT * FROM employees, departments
WHERE departments.department_id = employees.department_id;

-- 2) użyć operatora JOIN do połączenia tabel
SELECT * FROM employees JOIN departments
ON departments.department_id = employees.department_id;


-- JOIN oferuje więcej możliwości (LEFT JOIN, JOIN USING, NATURAL JOIN - to wszystko zaraz opowiemy)
-- składniowo JOIN jest częścią klauzuli FROM
-- przekonamy się pisząc zapytanie łączące 3 tabele
-- dodając warunek filtrowania

-- 1) Warunki dotyczące złączenia oraz zwykłe warunki logiczne są razem w WHERE
SELECT *
FROM employees, departments, locations
WHERE departments.department_id = employees.department_id
    AND locations.location_id = departments.location_id
    AND salary >= 10000;

-- 2) Po kolei dodajemy tabel i dla każdej od razu po JOIN wpisujemy warunek złączenia,
--	natomiast w WHERE mamy tylko inne warunki logiczne
SELECT *
FROM employees
    JOIN departments ON departments.department_id = employees.department_id
    JOIN locations ON locations.location_id = departments.location_id
WHERE salary >= 10000;


-- Aliasy tabel
-- Zamiast powtarzać długie nazwy tabel:
SELECT *
FROM employees, departments, public.locations
WHERE departments.department_id = employees.department_id
  AND public.locations.location_id = departments.location_id;

-- ... można wprowadzić "aliasy tabel" w sekcji FROM:
SELECT *
FROM employees emp, departments d, public.locations l
WHERE d.department_id = emp.department_id
  AND l.location_id = d.location_id;

-- Alias to nasza lokalna nazwa, którą nadajemy tabeli w obrębie jednego zapytania.
-- Alias wprowadzamy we FROM, a używamy w warunkach i pozostałych klauzulach zamiast nazwy tabeli
-- (nie możemy już używać oryginalnej nazwy tabeli!).

-- Są sytuacje, gdy alias tabeli jest niezbędny, aby napisać jakieś zapytanie (np. tzw. self join).

-- Uwaga składniowa: w Oraclu nie wolno pisać AS przed aliasem tabeli (w PostgreSQL i MySQL można, ale lepiej pisać tak, aby było przenośnie...).

--* Możliwości JOIN.
-- Są trzy sposoby podawania warunku złączenia: JOIN ON, JOIN USING, NATURAL JOIN
-- Są też różne "kierunki" złączeń (jak postępować z wartościami niepasującymi): INNER, LEFT, RIGHT, FULL
-- Dodatkowo istnieje CROSS JOIN
-- Daje to 3×4+1 = 13 różnych wersji JOINa :-)


-- Przegląd 3 sposobów podawania warunku złączenia
-- 1) tabela1 JOIN tabela2 ON warunek_logiczny
SELECT *
FROM employees e JOIN jobs j ON j.job_id = e.job_id;

-- 2) tabela1 JOIN tabela2 USING(kolumny)
-- Dobierane są takie rekordy, które mają jednakową wartość w tej kolumnie, która jest wymieniona w USING.
-- (teoretycznie może być podanych więcej kolumn)
SELECT *
FROM employees JOIN jobs USING(job_id);

-- 3) tabela1 NATURAL JOIN tabela2
-- Automatycznie znajdowane są kolumny, które mają jednakową nazwę w obu tabelach,
-- i wybierane są te kombinacje rekordów, gdzie wartości tych kolumn są równe.
SELECT *
FROM employees NATURAL JOIN jobs;

-- Dodatkowe szczegóły na temat JOIN ON / JOIN USING / NATURAL JOIN

-- NATURAL JOIN działa poprawnie tylko wtedy, gdy kolumna, po której łączymy,
-- ma identyczną nazwę i jest jedyną kolumną o wspólnej nazwie w tych tabelach.
SELECT *
FROM employees NATURAL JOIN departments;

-- Oprócz kolumny department_id istnieje też kolumna manager_id, która:
-- w tabeli employees mówi, kto jest szefem danej osoby
-- w tabeli departments mówi, kto jest szefem departamentu
-- to zapytanie jest równoważne:
-- (wynik zawiera tylko te osoby, których szef jest jednocześnie szefem departamentu)
SELECT *
FROM employees JOIN departments USING(department_id, manager_id);

-- Z kolei gdy nie ma żadnej wspólnej kolumny, to NATURAL JOIN też działa,
-- ale w wyniku daje wszystkie kombinacje, jak CROSS JOIN
SELECT *
FROM departments NATURAL JOIN countries;


-- JOIN ON jest najbardziej ogólny - pozwala wpisać dowolny warunek logiczny.
-- przydaje się szczególnie, gdy:
-- * warunek jest inny niż prosta równość; np. dobierając dane musimy zignorować wielkość liter,
--   trzeba wyciągnąć pole daty, wyciąć fragment napisu, albo porównać "w przybliżeniu"
-- * kolumna, po której łączymy tabele, ma różne nazwy w obu tabelach

-- Sztuczny przykład - połącz osoby z miastami tak, aby pierwsza miasta i nazwiska litera się zgadzała
SELECT first_name, last_name, city
FROM employees LEFT JOIN locations ON substr(last_name, 1, 1) = substr(city, 1, 1)
ORDER BY employee_id;


-- Utożsamianie kolumn w USING i NATURAL JOIN

-- Gdy używamy przecinka i WHERE albo JOIN ON, to w wynikach znajdą się kolumny z obu tabel.
-- Jeśli istnieją kolumny o tej samej nazwie, to obie będą obecne jako osobne kolumny.
-- Gdybyśmy teraz chcieli wskazać taką kolumnę w części WHERE, GROUP BY, ORDER BY albo SELECT, to musimy zrobić to poprzez alias lub nazwę tabeli.
-- Poniższe zapytanie jest błędne: nazwa job_id jest niejednoznacza:
SELECT *
FROM employees e JOIN jobs j ON j.job_id = e.job_id
WHERE job_id = 'ST_CLERK';

-- To jest OK:
SELECT *
FROM employees e JOIN jobs j ON j.job_id = e.job_id
WHERE j.job_id = 'ST_CLERK';

-- Gdy używamy USING albo NATURAL JOIN, to w wynikach znajdzie się tylko jeden egzemplarz kolumny, wg której łączymy.

-- Uwaga dot. Oracla, ale nie dotyczy Postgresa! :
-- W Oracle NIE MOŻNA odwoływać się tej kolumny poprzez alias/nazwę tabeli, można tylko bez prefiksu.
-- Poniższe zapytanie jest błędne w Oracle, niepotrzebny kwalifikator (prefiks)
SELECT *
FROM employees e JOIN jobs j USING(job_id)
WHERE j.job_id = 'ST_CLERK';

-- Błędny w Oracle jest również taki zapis
-- gdzie próbujemy pobrać wszystkie kolumny z jednej z tabel za pomocą tabela.*
SELECT e.*, job_id, j.job_title
FROM employees e JOIN jobs j USING(job_id);

-- Ale oba te zapytania zadziałały w PostgreSQL.

-- To jest OK w obu bazach:
SELECT *
FROM employees e JOIN jobs j USING(job_id)
WHERE job_id = 'ST_CLERK';



-- 4 "kierunki" złączeń
-- wprowadzenie w temat:
-- istnieje osoba, która nie ma ustawionego department_id
SELECT * FROM employees WHERE department_id IS NULL;
-- istnieją departamenty, w których nikt nie pracuje
SELECT * FROM departments
WHERE department_id NOT IN (SELECT DISTINCT department_id
                            FROM employees
                            WHERE department_id IS NOT NULL);

SELECT department_id FROM departments
EXCEPT SELECT department_id FROM employees;


-- INNER JOIN - złączenie wewnętrzne,
-- w wynikach pojawią się tylko te rekordy z lewej i prawej tabeli, dla których znaleziono dopasowanie po drugiej stronie
-- Tutaj 106 wyników.
-- Nie ma Kimberely Grant, bo ona ma NULLa w departamencie
-- Nie ma departamentów, w których nikt nie pracuje, np. Payroll.
SELECT *
FROM employees e INNER JOIN departments d ON d.department_id = e.department_id
ORDER BY e.employee_id, d.department_id;

-- słowo INNER jest opcjonalne, i gdy napiszemy samo JOIN , to zadziała tak, jak przy słowie INNER
SELECT *
FROM employees e JOIN departments d ON d.department_id = e.department_id
ORDER BY e.employee_id, d.department_id;

-- Pozostałe 3 rodzaje złączeń nazywane są złączeniami zewnętrznymi: LEFT, RIGHT i FULL
-- Za każdym z tych słów można dopisać OUTER, ale to nie wpływa na działanie.
-- LEFT JOIN - pojawią się wszystkie rekordy z lewej tabeli,
-- a z prawej tylko te, które zostały do czegoś dopasowane.
-- Tutaj w wynikach jest Kimberely Grant, ale nie ma nadmiarowych departamentów.
SELECT *
FROM employees e LEFT JOIN departments d ON d.department_id = e.department_id
ORDER BY e.employee_id, d.department_id;

-- OUTER nic nie znaczy, ale można sobie napisać
SELECT *
FROM employees e LEFT OUTER JOIN departments d ON d.department_id = e.department_id
ORDER BY e.employee_id, d.department_id;


-- RIGHT JOIN - pojawią się wszystkie rekordy z prawej tabeli,
-- a z lewej tylko te, które zostały do czegoś dopasowane.
-- Tutaj w wynikach nie ma Kimberely Grant, ale są te dodatkowe departamenty.
SELECT *
FROM employees e RIGHT JOIN departments d ON d.department_id = e.department_id
ORDER BY e.employee_id, d.department_id;

-- FULL JOIN - pojawią się wszystkie rekordy z lewej oraz prawej tabeli, połączone, jeśli się da.
-- Tutaj będzie zarówno K.Grant, jak i dodatkowe departamenty.
SELECT *
FROM employees e FULL JOIN departments d ON d.department_id = e.department_id
ORDER BY e.employee_id, d.department_id;

-- to samo!
SELECT *
FROM employees e FULL OUTER JOIN departments d ON d.department_id = e.department_id
ORDER BY e.employee_id, d.department_id;

-- dziwny przykład, aby pokazać kolejność słów kluczowych:
SELECT * FROM countries NATURAL FULL OUTER JOIN locations;

-- Poza 12 kombinacjami powyższych ustawień, mamy jeszcze CROSS JOIN / "złączenie krzyżowe",
-- czyli "iloczyn kartezjański" (Cartesian product)
-- kombinacja każdego rekordu z lewej tabeli z każdym rekordem z prawej; działa jak przecinek we FROM
SELECT first_name, last_name, city
FROM employees CROSS JOIN locations
ORDER BY 2, 1, 3;

SELECT 'Pracowniku ' || first_name || ' ' || last_name || ', jedziesz w delegację do ' || city
FROM employees CROSS JOIN locations
ORDER BY 1;


-- To, co wpisujemy jako elementy składowe JOIN, to są tzw "common table expressions".
-- W pewnych sytuacjach można brać je w nawiasy i zmieniać strukturę.
SELECT * FROM
    (locations
        LEFT JOIN countries USING(country_id)
        LEFT JOIN regions USING(region_id))
    RIGHT JOIN
        (employees
            LEFT JOIN departments USING(department_id))
    ON locations.location_id = departments.location_id
ORDER BY employee_id;

-- Mini - ćwiczenia
-- Wypisz imię i nazwisko tych osób, które pracują w mieście Oxford
SELECT first_name, last_name
FROM employees
    JOIN departments USING(department_id)
    JOIN locations USING(location_id)
WHERE city = 'Oxford';

-- Przykład na zawodnikach:
-- Wypisz zawodników oraz ich trenerów łącząc na podstawie kraju
SELECT * FROM zawodnicy;
SELECT * FROM trenerzy;

SELECT * FROM zawodnicy JOIN trenerzy USING(kraj);
-- nie każdy zawodnik ma trenera, i nie każdy trener ma zawodników :)
SELECT * FROM zawodnicy FULL JOIN trenerzy USING(kraj);


/* Połącz tabele employees, jobs, departments, locations, countries, regions
   tak, aby pojawiły się dane wszystkich pracowników.
   Używając operatora || do łączenia tekstów,
   stwórz zapytanie, które zwraca dane adresowe pracownika.
   Kolejne kolumny to mają być tak jakby kolejne wiersze do wydrukowania na kopercie z korespondencją służbową.
*/
SELECT employee_id AS "id"
    , first_name || ' ' || last_name AS "Pracownik"
    , job_title AS "Stanowisko"
    , department_name AS "Departament"
    , street_address AS "Ulica"
    , postal_code || ', ' || city AS "Miasto"
    , country_name AS "Kraj"
FROM employees
    LEFT JOIN jobs USING(job_id)
    LEFT JOIN departments USING(department_id)
    LEFT JOIN locations USING(location_id)
    LEFT JOIN countries USING(country_id)
    -- LEFT JOIN regions USING(region_id)
ORDER BY employee_id;

/* Kolumna manager_id w pracowniku mówi, kto jest jego szefem (podane jest id tego szefa).
   Wypisz
   - imię i nazwisko każdego pracownika X
   - imię i nazwisko jego szefa
   - nazwę departamentu, w którym pracuje pracownik X
   - imię i nazwisko szefa tego departamentu
*/
SELECT * FROM employees;

SELECT e1.employee_id, e1.first_name, e1.last_name, e1.manager_id,
 e2.employee_id, e2.first_name, e2.last_name
FROM employees e1 LEFT JOIN employees e2 ON e2.employee_id = e1.manager_id;

SELECT e.employee_id
    , e.first_name || ' ' || e.last_name AS "Pracownik"
    , m.first_name || ' ' || m.last_name AS "Szef"
    , d.department_name AS "Departament"
    , dm.first_name || ' ' || dm.last_name AS "Szef departamentu"
FROM employees e
    LEFT JOIN departments d ON e.department_id = d.department_id
    LEFT JOIN employees m ON e.manager_id = m.employee_id
    LEFT JOIN employees dm ON d.manager_id = dm.employee_id
ORDER BY e.employee_id;

-- Inne sposoby odczytywania zależności hierarchicznych...
CREATE EXTENSION IF NOT EXISTS tablefunc;

SELECT * FROM connectby('employees', 'employee_id', 'manager_id', 'employee_id', '100', 0, '/')
AS managers(keyid int, parent_keyid int, level int, aaa text, pos int)
;

