--* ORDER BY - sortowanie wyników *--
SELECT * FROM employees
ORDER BY salary;

-- czym może być kryterium sortowania?
-- można podać nazwę kolumny, można wyrażenie
SELECT * FROM employees
ORDER BY length(last_name);

-- Jeśli w SELECT tworzymy kolumnę i nadaliśmy jej alias, to w OREDER BY można podać ten alias
SELECT first_name || ' ' || last_name AS "Kto"
    , 12 * salary AS "Roczne zarobki"
FROM employees
ORDER BY "Roczne zarobki";

SELECT first_name || ' ' || last_name AS "Kto"
    , 12 * salary AS roczne
FROM employees
ORDER BY roczne;

-- Można po prostu podać nr kolumny wynikowej, po której sortujemy, licząc od 1
SELECT first_name, last_name, 12 * salary AS "Roczne zarobki", manager_id
FROM employees
ORDER BY 3;

-- Domyślnie kolejność jest rosnąca
-- aby była malejąca, za kryterium sortowania należy dopisać DESC
SELECT * FROM employees
ORDER BY salary DESC;

-- Analogicznie istnieje słowo ASC , ale raczej się go nie używa, bo kolejność rosnąca jest domyślna
SELECT * FROM employees
ORDER BY salary ASC;

-- Może być kilka kryteriów sortowania
-- po pierwsze wg nazwiska, po drugie wg imienia
SELECT * FROM employees
ORDER BY last_name, first_name;

-- W przypadku kilku kryteriów, każde z nich domyślnie jest rosnąco,
-- przy każdym malejącym trzeba osobno pisać DESC
-- Tutaj: rosnąco wg miasta, nast malejąco wg pensji, rosnąco wg nazwisko i imion
-- Zamiast tabeli employees, uzywamy "widoku" emp_detils_view, który łaczy dane z kilku tabel.
SELECT employee_id, first_name, last_name, job_title, salary, department_name, city
FROM emp_details_view
ORDER BY city, salary DESC, last_name, first_name;

SELECT first_name || ' ' || last_name AS "Osoba"
    , 12 * salary AS "Roczne zarobki"
    , extract(YEAR FROM hire_date) AS "Rok zatrudnienia"
FROM employees
ORDER BY 3, 2 DESC;

-- Zobaczmy gdzie znajdą się NULL-e podczas sortowania.
-- Domyślnie NULLe są traktowane tak, jakby były większe od zwykłych wartości.
-- (tak jest w PostgreSQL i Oracle, zgodnie ze standardem; w MySQL i SQLite jest odwrotnie)

-- Przy ASC pójdą na koniec
SELECT * FROM employees
ORDER BY commission_pct;

-- Przy DESC pójdą na początek
SELECT * FROM employees
ORDER BY commission_pct DESC;

-- Gdy na końcu kryterium sortowania dopiszemy NULLS FIRST lub NULLS LAST, sami określamy ich położenie
SELECT * FROM employees
ORDER BY commission_pct DESC NULLS LAST;

SELECT * FROM employees
ORDER BY commission_pct ASC NULLS FIRST;

SELECT * FROM employees
ORDER BY commission_pct NULLS FIRST;



--* LIMIT / OFFSET *--
-- W różnych bazach danych istnieją różne sposoby, aby zapytanie zwróciło tylko pierwszy rekord / X pierwszych rekordów.
-- W PostgreSQL od dawna służy do tego dodatkowa klauzula LIMIT / OFFSET dopsywana na końcu zapytania (za klauzulą ORDER BY).

-- To zwraca pierwsze 10 rekordów
SELECT * FROM employees
ORDER BY salary DESC
LIMIT 10;

-- Używając OFFSET możemy pobierać następne grupy, np. "trzecią dziesiątkę"
-- OFFSET pomija określoną liczbę rekordów i zaczyna zwracać pewną liczbę następnych
SELECT * FROM employees
ORDER BY salary DESC
LIMIT 10 OFFSET 20;

-- Od rekordu 91 do samego końca:
SELECT * FROM employees
ORDER BY salary DESC
OFFSET 90;

-- Używanie LIMIT / OFFSET bez sortowania jest poprawne technicznie,
-- ale ma mały sens logiczny.
-- Chyba, że w technicznym celu pobrania "próbki danych"...
SELECT * FROM employees
LIMIT 5;

-- Potecjalnym zastosowaniem jest odczyt jednego rekordu o wartości maksymalnej lub minimalnej.
SELECT * FROM employees
ORDER BY salary
LIMIT 1;

-- Alternatywnym rozwiązaniem jest użycie podzapytania:
SELECT * FROM employees
WHERE salary = (SELECT min(salary) FROM employees);

-- Ale w konstrukcji LIMIT kryje się niejednoznaczność w sytuacji, gdy istnieje wiele rekordów o jednakowej wartości.
-- Przykład: istnieją 2 osoby zarabiające 17 tys. Zajmują ex-equo 2 miejsce w firmie.
SELECT * FROM employees
ORDER BY salary DESC
LIMIT 2;



--* Konstrukcja FETCH *--
-- Standard SQL wprowadził rozwiązanie "FETCH", które jest obecnie obsługiwane m.in. przez Oracle i PostgreSQL.
-- W porównaniu do LIMIT/OFFSET dodatkową możliwością jest pobieranie wszystkich rekordów, których wartość
-- znajduje się na granicy - dopisek "WITH TIES".

-- Służy do pobrania tylko określonej liczby rekordów. Umieszcza się ją na samym końcu.
SELECT * FROM employees
FETCH NEXT 10 ROWS ONLY;

-- Jeśli wyniki nie są sortowane, to nie mamy gwarancji, które dokładnie rekordy odczytamy w ten sposób.
-- To podejście może być przydatne, gdy chcemy tylko sprawdzić jak wyglądają dane.

-- Najczęściej jednak FETCH używa się w połączeniu z ORDER BY.
-- To zapytanie zwraca 10 najbogatszych osób w firmie
SELECT * FROM employees
ORDER BY salary DESC
FETCH FIRST 10 ROWS ONLY;

-- Kwestia równych wartości...
-- To zapytanie ma zwrócić 2 najbogatsze osoby w firmie
SELECT * FROM employees
ORDER BY salary DESC
FETCH FIRST 2 ROWS ONLY;

-- Aby odczytać wszystkich, którzy maja tyle samo, można użyc WITH TIES
-- Postgres będzie wiedział, ze ma doczytać kolejne rekordy z taka sama wartością, po której sortowaliśmy
SELECT * FROM employees
ORDER BY salary DESC
FETCH FIRST 2 ROWS WITH TIES;

-- na pewno zwróci jeden rekord
SELECT * FROM employees
ORDER BY salary
FETCH FIRST 1 ROW ONLY;

-- można pominąć liczbę 1, jeśli chcemy odczytać jeden rekord
SELECT * FROM employees
ORDER BY salary
FETCH FIRST ROW ONLY;

-- teoretycznie może być więcej osób, które zarabiają minimum - WITH TIES zwróciłoby wszystkie
SELECT * FROM employees
ORDER BY salary
FETCH FIRST ROW WITH TIES;

-- Na działanie nie wpływa to, czy napiszemy ROW czy ROWS ani FIRST czy NEXT
-- To też działa, chociaż głupio wygląda:
SELECT * FROM employees
ORDER BY salary
FETCH NEXT ROWS ONLY;


-- Jeśli dodamy jeszcze opcję OFFSET, to zaczniemy pobierać rekordy nie od pierwszego, tylko od N-tego
-- Wtedy bardziej czytelne będzie użycie słowa NEXT zamiast FIRST, chociaż nie wpływa to na działanie.

-- Właśnie tego typu konstrukcji używają programiści do realizacji "stronicowania".

-- rekordy od 21 do 30
SELECT * FROM employees
ORDER BY salary DESC
OFFSET 20 ROWS
FETCH NEXT 10 ROWS ONLY;

-- W PostgreSQL nie ma FETCH NEXT 20 PERCENT

-- diagramy:
-- https://docs.oracle.com/en/database/oracle/oracle-database/19/sqlrf/SELECT.html 
