--* SELECT *--

-- SELECT służy do odczytu danych
-- Wyniki zasadniczo mają kształt tabeli,
-- ale w szczególnych przypadkach mogą wyjść: pojedynczy wiersz, pojedyncza kolumna, pojedyncza wartość (tzw. zapytanie skalarne)
-- zapytanie może też zwrócić puste wyniki

SELECT * FROM employees;
SELECT first_name, last_name FROM employees;
SELECT first_name, last_name FROM employees WHERE salary > 50000;

SELECT avg(salary) FROM employees;

/* Ogólna składnia polecenia SELECT:
SELECT kolumny
FROM tabele
WHERE warunek rekordów
GROUP BY kryteria grupowania
HAVING warunek grup
ORDER BY kryteria sortowania
...

Poszczególne części nazywa się "kaluzulami" (clause).

Oprócz tej podstawy istnieją dodatkowe mniej lub bardziej przenośne klauzule, jak
np. LIMIT / OFFSET w PostgreSQL
FETCH, WITH, UNION itp, WINDOW....
Można też używać "podzapytań".
*/


-- W większości baz danych, ale nie w Oracle, możn a napisać samego selecta bez dodatkowych klauzul.
SELECT 2*3;
SELECT current_timestamp;
-- W Oracle: SELECT 2*3 FROM dual;

--* Klauzula SELECT *--
-- Sama klauzula SELECT określa kolumny, które mają znaleźć się w wyniku.

-- Można podać te kolumny tabeli źródłowej, tóre mają trafić do wyniku.
SELECT first_name, last_name, salary
FROM employees;

-- W klauzuli SELECT można nie tylko wskazywać istniejące kolumny,
-- ale można też wpisywać wyrażenia, które coś obliczają.
-- Przy okazji: || w SQL oznacza łączenie tekstów
SELECT 'Pracownik ' || first_name || ' ' || last_name, 12 * salary, current_timestamp
FROM employees;

SELECT concat('Pracownik ', first_name, ' ', last_name), 12 * salary, current_timestamp
FROM employees;

-- Wynikowym kolumnom można nadać własne nazwy, tzw "aliasy kolumn".
-- Nazwa bez cudzysłowów nie może zawierać spacji ani wielu innych znaków specjalnych
-- jest automatycznie zamieniana na małe litery.
SELECT first_name || ' ' || last_name AS kto
    , 12 * salary AS roczne_zarobki
    , current_date - hire_date AS Staz
FROM employees;


-- W aliasach kolumn bardzo często stosuje się "nazwy cytowane".
-- Wtedy można użyć spacji i innych znaków, a wielkość liter jest zachowywana.
SELECT first_name || ' ' || last_name AS "Kto"
    , 12 * salary AS "Roczne zarobki"
    , current_date - hire_date AS "Staż pracy"
FROM employees;

-- Tworząc alias kolumny nie musimy pisać słowa AS,
-- ale wersja z AS jest bardziej przenośna (MS SQL Server)
SELECT first_name || ' ' || last_name Kto
    , 12 * salary "Roczne zarobki"
    , current_date - hire_date "Staż pracy"
FROM employees;


-- Jeśli podzapytanie wprowadza aliasy kolumn,
-- to zapytanie zewnętrzne musi używać tych aliasów jako nazw kolumn,
-- a oryginalne nazwy nie są dostępne.
SELECT imie, nazwisko, zarobki AS "Zarobki", "Roczne zarobki"
FROM (SELECT first_name AS imie, last_name AS nazwisko
        , salary AS zarobki
        , 12 * salary AS "Roczne zarobki"
    FROM employees) podzapytanie
WHERE "Roczne zarobki" >= 100000;


-- * - weź wszystkie dostępne kolumny
SELECT * FROM employees;

-- <zapytania logicznie bez sensu>
SELECT * FROM locations, countries;

SELECT locations.*, regions.* FROM locations, countries, regions;
-- nie ma country_name
-- </zapytania logicznie bez sensu>

-- Jeśli chcę odczytać wszystkie kolumny z tabeli, a oprócz nich dodać kolejne (wyliczane wyrażeniem):
SELECT *, lower(email) || '@alx.pl' AS prawdziwy_email
FROM employees;

-- Powyższe działa w PostgreSQL, ale nie Oracle.
-- W obu bazach zadziała taka wersja:
SELECT employees.*, lower(email) || '@alx.pl' AS prawdziwy_email
FROM employees;


-- DISTINCT powoduje pobranie danych bez powtórzeń:
SELECT first_name FROM employees;

SELECT first_name FROM employees ORDER BY 1;

SELECT DISTINCT first_name FROM employees;
SELECT DISTINCT first_name FROM employees ORDER BY first_name;


-- Jeśli zapytanie zwraca kilka kolumn, to unikalne mają być "krotki", czyli kombinacje wartości.
-- Np. teraz niektóre imiona się powtarzają, niektóre nazwiska się powtarzają,
-- ale nie ma osoby o identycznej kombinacji imienia i nazwiska.
SELECT DISTINCT first_name, last_name FROM employees;

-- Spróbujmy na inicjałach, aby faktycznie pojawiły się powtórzone pary.
-- Teraz sa powtorzenia
SELECT substr(first_name, 1, 1), substr(last_name, 1, 1) FROM employees;

-- Teaz moga powtarzac sie pierwsze litery imion, moga nazwisk, ale nie powtarzaja sie pary.
SELECT DISTINCT substr(first_name, 1, 1), substr(last_name, 1, 1) FROM employees;


-- Dodatkowy temat - "lista imion"
SELECT first_name FROM employees;
SELECT string_agg(first_name, ', ') FROM employees;


--* WHERE i warunki logiczne *--

-- W PostgreSQL istnieje typ boolean (wartości logicznych)
-- W PostgreSQL (inaczej niż m.in. w Oracle) można wynik porównanie czy innego sprawdzenia wypisać jako jedną z kolumn wynikowych:
SELECT first_name, last_name, salary, salary >= 10000 AS bogaty FROM employees;

-- Warunków logicznych najczęściej używa się do przefiltrowania rekordów.
-- W wynikach zapytania z klauzulą WHERE znajdą tylko te rekordy, które spełniają warunek.
SELECT first_name, last_name, salary
FROM employees
WHERE salary >= 10000;

-- W warunku można odwołać się także do kolumn, których się nie zwraca w wyniku
SELECT first_name, last_name
FROM employees
WHERE job_id = 'IT_PROG';

-- operatory porównania = < <= > >=
-- nierówne na dwa sposoby: !=  <>
SELECT first_name, last_name
FROM employees
WHERE job_id != 'IT_PROG';


-- Spójniki logiczne
-- AND - koniunkcja - oba warunki muszą być spełnione
-- OR - alternatywa - co najmniej jeden warunek musi być spełniony
-- NOT - negacja
SELECT * FROM employees WHERE salary = 9000;
SELECT * FROM employees WHERE job_id = 'IT_PROG';

-- AND : programiści, którzy zarabiają 9000
SELECT * FROM employees WHERE job_id = 'IT_PROG' AND salary = 9000;

-- OR : wszyscy programiści oraz wszyscy, którzy zarabiają 9000
-- Alexander Hunold pojawia się tylko raz
SELECT * FROM employees WHERE job_id = 'IT_PROG' OR salary = 9000;

-- Wskazówka dot. formatownia kodu, gdy jest dużo warunków składowych
SELECT employee_id
    , first_name || ' ' || last_name AS "Kto"
    , 12 * salary AS "Roczne zarobki"
    , current_date - hire_date AS "Staż pracy"
FROM employees
WHERE 1=1
  AND salary >= 5000
  AND job_id = 'SA_REP'
ORDER BY employee_id;


-- Aby sprawdzić, czy wartość mieści się w przedziale, możemy:
-- 1) użyć dwóch porówań połączonych AND
SELECT * FROM employees WHERE salary >= 5000 AND salary <= 10000;

-- 2) za pomocą BETWEEN
-- uwaga: BETWEEN jest obustronnie domknięty
SELECT * FROM employees WHERE salary BETWEEN 5000 AND 10000;


-- Odczytajmy pracowników ze stanowisk ST_CLERK oraz ST_MAN
-- sposób 1: spójnik OR
SELECT * FROM employees WHERE job_id = 'ST_CLERK' OR job_id = 'ST_MAN';

-- sposób 2: operator IN - czy wartość należy do podanego zbioru
SELECT * FROM employees WHERE job_id IN ('ST_CLERK', 'ST_MAN');

-- Ale operator IN może być też używany w połączeniu z podzapytaniem...
-- Zakładając, że posiadamy tabelę selected_jobs, można by tak:
/* SELECT * FROM employees
   WHERE job_id IN (SELECT job FROM selected_jobs);
*/

-- sposób 3: operator LIKE (tylko gdy joby mają część wspólną)
-- wszystkie rekordy, gdzie job_id zaczyna się na ST
SELECT * FROM employees WHERE job_id LIKE 'ST%';

-- Po lewej stronie LIKE znajduje się zwykły napis, zazwyczaj kolumna z tabeli.
-- Po prawej jest "wzorzec". We wzorcach:
-- znak % oznacza dowolnej długości ciąg dowolnych znaków
-- znak _ oznacza pojedynczy dowolny znak
-- za pomocą symbolu \ można "wy-escape-ować" znaki specjalne, gdyby miały być faktyczną treścią
-- Wielkość liter MA znaczenie.
SELECT * FROM employees WHERE job_id LIKE 'ST\_%';

-- wszystkie nazwiska zaczynające się na K
SELECT first_name, last_name FROM employees WHERE last_name LIKE 'K%';

-- pierwsza i ostatnia
SELECT first_name, last_name FROM employees WHERE last_name LIKE 'K%g';

-- druga litera nazwiska jest równa a
SELECT * FROM employees WHERE last_name LIKE '_a%';

-- Zadanie: wypisz te nazwy miast, które zawierają co najmniej 2 litery o
SELECT city FROM locations WHERE city LIKE '%o%o%';

-- Wyniki nie obejmują Oxford.

-- Jeśli chcemy, aby również duże 'O' zostało uwzględnione:
-- rozwiązanie ogólne (np. działające w Oraclu)
SELECT city FROM locations WHERE lower(city) LIKE '%o%o%'

-- rozwiązanie Postgresa - specjalna wersja LIKE o nazwie ILIKE
SELECT city FROM locations WHERE city ILIKE '%o%o%'

-- Jeszcze inna wersja bazująca na "wyrażeniach regularnych SQL", które są trochę inne, niż regexpy znane z innych platform
SELECT city FROM locations WHERE city SIMILAR TO '%(o%){2}';

SELECT city FROM locations WHERE regexp_like(city, '.*((o|O).*){2}');

-- Informacja nt SIMILAR TO oraz regexp_like
-- https://www.postgresql.org/docs/17/functions-matching.html

-- Generalnie porówania tekstu są case-sensitive

-- Aby zignorować wielkość liter
SELECT first_name, last_name
FROM employees
WHERE lower(job_id) = 'st_clerk';

-- PostgreSQL posiada też operator ILIKE, czyli wersję LIKE nie zwracającą uwagi na wielkość liter.
SELECT first_name, last_name
FROM employees
WHERE job_id ILIKE 'st\_clerk';

-- W WHERE nie wolno używać aliasów kolumn wprowadzonych w SELECT.
SELECT first_name || ' ' || last_name AS kto
    , 12 * salary AS roczne_zarobki
FROM employees
WHERE roczne_zarobki >= 100000;

SELECT first_name || ' ' || last_name AS kto
    , 12 * salary AS "roczne_zarobki"
FROM employees
WHERE "roczne_zarobki" >= 100000;

--* Temat NULL *--

-- Zobaczmy, że NULL nie jest zwykłą wartością, tylko powoduje dziwne zachowania, gdy się pojawia.

-- W tabeli employees istnieje kolumna commission_pct, która u niektórych pracowników zawiera liczbę,
-- a u pozostałych ma wartość NULL.
SELECT * FROM employees;
SELECT * FROM employees WHERE commission_pct > 0.2;
SELECT * FROM employees WHERE commission_pct <= 0.2;
-- Dla wartości NULL nieprawdą jest ani > , ani <
-- Ci, którzy nie uzyskują prowizji, nie pojawią się ani w jednej, ani w drugiej grupie.

-- Aby sprawdzić czy jest jest równa NULL, czy nie, należy użyć wyspecjalizowanych operatorów IS (NOT) NULL
SELECT * FROM employees WHERE commission_pct = NULL;
SELECT * FROM employees WHERE commission_pct IS NULL;

-- Inny problem - NULL w wyrażniu arytmetycznym powoduje, że cały wynik też jest nullem.
-- NULL to nie jest zero.
SELECT first_name || ' ' || last_name
    , salary + commission_pct * salary AS wyplata
    , substr(first_name, 1, 1) || '. ' || last_name || ' ma prowizję ' || 100*commission_pct || ' procent' AS opis
FROM employees;

-- rozwiązanie tego problemu:
SELECT first_name || ' ' || last_name
    , salary + coalesce(commission_pct, 0) * salary AS wyplata
    , substr(first_name, 1, 1) || '. ' || last_name ||
        coalesce(' ma prowizję ' || 100*commission_pct || ' procent', ' bez prowizji') AS opis
    , concat(substr(first_name, 1, 1), '. ', last_name, ' ', 100*commission_pct) AS opis2
FROM employees;


/* Praprzyczyną tego dziwnego zachowania jest fakt, iż na początku historii SQL uznano,
że NULL będzie oznaczał BRAK WIEDZY jaka jest wartość danego pola.
Czyli tak jakby w tym przykładzie wartość NULL znaczyła "nie wiadomo jaką prowizję uzyskuje pracownik".
Takie myślenie miałoby sens w takim prykładzie:
Osoba ma kolumnę data_urodzenia, ale dla pewnej osoby nie znamy jej daty urodzenia.
To wtedy wpisujemy NULL.

Do takiego znaczenia wartości NULL dostosowano całą logikę i działanie operatorów.

W praktyce jednak często przyjmuje się, że NULL oznacza "pustą wartość".
Oznacza "wiemy, że czegoś nie ma", jest puste, zerowe, brak.
Tak jak w tym przykładzie, intencją autora tabeli employees było na pewno,
aby NULL w polu commission_pct znaczył "pracownik nie otrzymuje żadnej prowizji".
I tak jest zazwyczaj w praktyce.
Jednak logika SQL tak tego nie rozumie. Dla niej NULL oznacza "nie wiadomo".

Więcej na temat teorii można szukać pod hasłem "logika trójwartościowa SQL".
*/

-- Porównanie z NULLem daje wynik NULL
SELECT 1 = NULL;

-- W warunkach SQL-owych obowiązuje "logika trójwartościowa": trye / false / null (nie wiadomo)
-- Spójniki logiczne starają się robić swoje nawet jeśli czasami jest nieznana wartość.
SELECT 1 = 1 AND 2 = NULL;
SELECT 1 = 0 AND 2 = NULL;
SELECT 1 = 1 OR 2 = NULL;

-- Operacje arytmetyczne itp. , gdzie wewnątrz pojawia się NULL, dają wynik NULL.
SELECT 2 * (3 + NULL) - 15 * 4;
SELECT 0 * NULL; -- jednak null

/* Gdy wykonujemy obliczenia, w których pojawia się wartość NULL,
   to zazwyczaj wynikiem całego wyrażenia też jest NULL („null zaraża”).

   Tutaj pewne uproszczenie: przyjmujemy, że prowizja jest liczona względem pensji,
   a nie sprzedaży (której w tej bazie danych nie ma).

   Dla każdego pracownika chcemy obliczyć jego wypłatę wraz z dodaną prowizją.
   Przy poniższym zapisie wynikiem w ostatniej kolumnie
   jest NULL u tych pracowników, którzy nie uzyskują prowizji.
*/
SELECT first_name, last_name, salary, commission_pct
    ,commission_pct * salary -- prowizja kwotowo
    ,salary + commission_pct * salary -- kwota do wypłaty
FROM employees;

-- Aby zastąpić wartość NULL inną "konkretną" wartością, można użyć funkcji COALESCE
SELECT first_name, last_name, salary, commission_pct
    ,coalesce(commission_pct * salary, 0) as "kwota prowizji"
    ,salary + coalesce(commission_pct, 0) * salary as "wypłata z prowizją"
FROM employees;

-- coalesce(wartosc1, wartosc2, ....) -- dowolnie wiele wartości po przecinku
-- zwraca pierwszą z wartości podanych jako parametry, która nie jest nullem (no chyba ze wszystkie są)
SELECT coalesce(null, null, 'bingo', null, 'kapusta');-- FROM dual;

-- coalesce jest częścią standardu SQL i działa w większości baz danych
-- W praktyce najczęściej zapisuje się tak: coalesce(kolumna_ktora_moze_byc_null, wartosc_domyslna)

-- Nie ma obowiązku podstawiania zera.
SELECT first_name, last_name, salary, commission_pct,
    coalesce(commission_pct, 1234)
FROM employees;


--! Specyfika Oracle !--

-- W Oraclu jest też funkcja nvl, która działa tak jak dwuargumentowy coalesce:
-- NVL(x, y):   jeśli x nie jest nullem, to zwraca x, a jeśli x jest nullem, to zwraca y
SELECT first_name, last_name, salary, commission_pct
    ,nvl(commission_pct * salary, 0) as "kwota prowizji"
    ,salary + nvl(commission_pct, 0) * salary as "wypłata z prowizją"
FROM employees;

-- Inne funkcje Oracla związane z obsługą NULL:

-- NVL2(x, y, z)
-- jeśli x nie jest NULL, to zwraca y
-- jeśli x jest NULLem, to zwraca z

SELECT first_name, last_name, commission_pct,
    nvl2(commission_pct, 'jest prowizja', 'brak prowizji')
FROM employees;

-- Na siłę można to wykorzystać do wyliczenia łącznej wypłaty:
SELECT first_name, last_name, salary, commission_pct,
    nvl2(commission_pct, salary*(1+commission_pct), salary) AS "wypłata"
FROM employees;

-- NULLIF(x, y) -- o!, to działa też w PostgreSQL
-- jeśli x = y, to wynikiem jest NULL
-- jeśli x != y, to wynikiem jest x
SELECT city, nullif(city, 'Toronto') FROM locations;



SELECT street_address, city, coalesce(state_province, '--BRAK DANYCH--') AS state_province
FROM locations;

-- Specyfika PostgreSQL i operatora ||
-- Gdy używamy || do łączenia napisów, a którykolwiek z argumentów jest NULL,
-- to cały wynik jest NULLem.
-- Funkcja concat traktuje nulle jak puste napisy.
SELECT city, state_province
    , city || ' ' || state_province AS "wersja_operator"
    , concat(city, ' ', state_province) AS "concat"
FROM locations;

-- W Oracle operator || nie ma problemów z nullami, gdyż w Oracle NULL jest utożsamiany z pustym tekstem.
-- '' i NULL to w Oracle to samo.
