package sklep.controller;

import java.util.List;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;

import jakarta.validation.Valid;
import sklep.model.Product;
import sklep.photo.PhotoUtil;
import sklep.repository.ProductRepository;
import sklep.utils.ExceptionUtils;

@Controller
@RequestMapping("/products_alt")
public class ProductControllerAlt {
    @Autowired
    private ProductRepository productRepository;
    
    @Autowired
    private PhotoUtil photoUtil;
    
    @GetMapping
    public String wszystkieProdukty(Model model) {
        List<Product> products = productRepository.findAll();
        model.addAttribute("products", products);
        return "products";
    }
    
    @GetMapping("/{id}")
    public String jedenProdukt(@PathVariable Integer id, Model model) {
        Optional<Product> product = productRepository.findById(id);
        if(product.isPresent()) {
            model.addAttribute("product", product.get());
            return "product";
        } else {
            model.addAttribute("product_id", id);
            return "missing_product";
        }
    }

    @GetMapping(path="/{id}/photo", produces="image/jpeg")
    @ResponseBody
    public byte[] foto(@PathVariable Integer id, Model model) {
        return photoUtil.readBytes(id);
    }
    
    @GetMapping("/new")
    public String nowyProdukt() {
        return "product_form_alt";
    }
    
    @GetMapping("/{id}/edit")
    public String edytujProdukt(@PathVariable Integer id, Model model) {
        Optional<Product> product = productRepository.findById(id);
        if(product.isPresent()) {
            model.addAttribute("product", product.get());
            return "product_form_alt";
        } else {
            model.addAttribute("product_id", id);
            return "missing_product";
        }
    }

    @PostMapping({ "/new", "/{id}/edit" })
    public String zapiszProdukt(Model model, @Valid Product product, BindingResult bindingResult) {
        // W tej wersji dane z wypełnionego formularza odbieramy w postaci jednego obiektu Product.
        // Spring sam wpisze dane do pól o takich samych nazwach.
        // Taki parametr od razu staje się częścią modelu (to jest tzw. ModelAttribute)
        System.out.println("Obiekt z formularza: " + product.getId() + " " + product.getProductName());
        
        // W tej wersji przed parametrem Product jest adnotacja @Valid
        // i dodatkowo do metoda posiada parametr BindingResult.
        // Wówczas Spring dokonuje walidacji obiektu przed wywołaniem tej metody
        // i informacje o wyniku walidacji przekazuje w parametrze BindingResult.
        // Metoda jest wywoływana zawsze, a to programista ma sprawdzić czy walidacja się powiodła.
        // W BindingResult znajdują się też informacje o błędach.
        if(bindingResult.hasErrors()) {
            model.addAttribute("errors", bindingResult.getAllErrors());
            model.addAttribute("fieldErrors", bindingResult.getFieldErrors());
//			FieldError fieldError = bindingResult.getFieldError("vat");
//			System.out.println(fieldError.getDefaultMessage());
        } else try {
            productRepository.save(product);
            System.out.println("Obiekt po zapisie: " + product.getId() + " " + product.getProductName());
            model.addAttribute("saved", true);
        } catch (Exception e) {
            // to jest obsługa błędów, które pojawiłyby się na etapie zapisywania do bazy danych
            // czyli błędy walidacji odkrywane dopiero przez Hibernate (gdyby nie było @Valid)
            // oraz błędy zgłaszane przez Postgresa
            model.addAttribute("errors", ExceptionUtils.allMessages(e));
        }
        
        return "product_form_alt";
    }
}
