package sklep.controller;

import java.math.BigDecimal;
import java.util.List;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.SessionAttribute;

import jakarta.validation.Valid;
import sklep.basket.Basket;
import sklep.model.Product;
import sklep.repository.ProductRepository;

@Controller
@RequestMapping("/products")
public class ProductController {
    @Autowired
    private ProductRepository productRepository;
    
    @GetMapping(produces="text/html")
    public String getProducts(Model model) {
        List<Product> products = productRepository.findAll();
        model.addAttribute("products", products);
        return "products";
    }
    
    @GetMapping("/szukaj")
    public String findProducts(Model model, String name, BigDecimal min, BigDecimal max) {
        List<Product> products = List.of();
        if(name != null && !name.isEmpty() && min == null && max == null) {
            products = productRepository.findByProductNameContainsIgnoringCase(name);
        } else if ((name == null || name.isEmpty()) && (min != null || max != null)) {
            if(min == null) {
                min = BigDecimal.ZERO;
            }
            if(max == null) {
                max = BigDecimal.valueOf(1000_000_000);
            }
            products = productRepository.findByPriceBetween(min, max);
        } else if (name != null && !name.isEmpty()) {
            if(min == null) {
                min = BigDecimal.ZERO;
            }
            if(max == null) {
                max = BigDecimal.valueOf(1000_000_000);
            }
            products = productRepository.findByProductNameContainingIgnoringCaseAndPriceBetween(name, min, max);
        }

        model.addAttribute("products", products);
        return "wyszukiwarka";
    }
    
    @GetMapping(path="/{id}", produces="text/html")
    public String getOneProduct(Model model, @PathVariable("id") Integer id) {
        Optional<Product> optProduct = productRepository.findById(id);
        if(optProduct.isPresent()) {
            model.addAttribute("product", optProduct.get());
            return "product";
        } else {
            return "missing_product";
        }
    }
    
    @GetMapping("/{id}/add-to-basket")
    public String addToBasket(@PathVariable("id") Integer productId,
                              @SessionAttribute Basket basket) {
        Optional<Product> product = productRepository.findById(productId);
        if(product.isPresent()) {
            basket.addProduct(product.get());
        } else {
            System.err.println("Nieznany produkt dodawany do koszyka: " + productId);
        }
        return "redirect:/products";
    }
    
    @GetMapping("/new")
    public String nowyProdukt(@ModelAttribute Product product) {
        return "product_form";
    }

    @GetMapping("/{id}/edit")
    public String edytujProdukt(@PathVariable int id, Model model) {
        Optional<Product> product = productRepository.findById(id);
        if(product.isPresent()) {
            model.addAttribute("product", product.get());
            return "product_form";
        } else {
            model.addAttribute("product_id", id);
            return "missing_product";
        }
    }

    @PostMapping({"/new", "/{id}/edit"})
    // Ta metoda zapisuje dane przysłane z formularza obojętnie, czy to było edit, czy new
    // Adnotacja @Valid powoduje, że Spring dokona walidacji obiektu PRZED uruchomieniem tej metody.
    // Jeśli nie ma dodatkowego parametru BindingResult, a są błędy walidacji, to Spring naszej metody w ogóle nie uruchomi.
    // Jeśli jednak metoda ma parametr BindingResult, to
    // metoda zawsze jest uruchamiana przez Springa,
    // a w tym parametrze zawarte są informacje o przebiegu walidacji, w tym błędy.
    // Bez adnotacji @ModelAttribute też działało
    public String saveProduct(@ModelAttribute @Valid Product product,
                              BindingResult bindingResult) {
        // W tej wersji dane z wypełnionego formularza odbieramy w postaci jednego obiektu Product.
        // Spring sam wpisze dane do pól o takich samych nazwach.
        // Taki parametr od razu staje się częścią modelu (to jest tzw. ModelAttribute)

        if(bindingResult.hasErrors()) {
            System.err.println("Są błędy: " + bindingResult.getAllErrors());
            // normalnie wyświetlilibyśmy coś na stronie...
            // model.addAttribute("errors", bindingResult.getAllErrors());
            // ale robi to za nas tag f:form i f:errors
            // Dopóki są błędy, pozostajemy na stronie formularza.
            return "product_form";
        } else {
            // Gdy podczas próby zapisu (operacja save) obiekt nie spełnia warunków walidacji, to jest wyrzucany wyjątek.
            System.out.println("id przed zapisem: " + product.getId());
            productRepository.save(product);
            System.out.println("id po zapisie: " + product.getId());
            return "redirect:/products";
            // Po zaakceptowaniu danych, klient jest przenoszony do listy produktów.
        }
    }
    
    @GetMapping(produces="text/plain")
    @ResponseBody
    public String getProductsTxt() {
        StringBuilder sb = new StringBuilder("Lista produktów:\n");
        List<Product> products = productRepository.findAll();
        for(Product product : products) {
            sb.append(" * ").append(product.getProductName())
              .append(" za cenę ").append(product.getPrice())
              .append('\n');
        }
        return sb.toString();
    }
}
