package org.example.demo;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ResponseBody;

@Controller
public class TimeController {

    @GetMapping("/time1")
    @ResponseBody
    public LocalTime time1() {
        // Gdy metoda zwraca jakiś obiekt (klasy innej niż String i niektóre inne "magiczne" klasy samego Springa),
        // a metoda posiada adnotację @ResponseBody, to dane odsyłane są w odpowiedzi w formacie JSON.
        return LocalTime.now();
    }

    @GetMapping("/time2")
    @ResponseBody
    public String time2() {
        return LocalTime.now().toString();
    }
    // Gdy metoda zwraca String i posiada adnotację @ResponseBody, to odsyłany jest tekst.
    // Domyślnie content-type to jest text/plain, ale jeśli przeglądarka żąda html, to Spring zwróci to oznaczone typem text/html
    // Aby wymusić wysłanie odpowiedniego nagłówka Content-Type, możemy dopisać parametr produces do adnotacji GetMapping lub RequestMapping
    
    private static final DateTimeFormatter FORMAT_DATY = DateTimeFormatter
            .ofPattern("EEEE, dd.MM.yyyy, 'godzina' HH:mm:ss", Locale.of("pl", "PL"));

    @GetMapping(path="/time3", produces="text/plain")
    @ResponseBody
    public String time3() {
        return LocalDateTime.now().format(FORMAT_DATY);
    }

    // Jak wysłać w odpowiedzi HTML?
    // 1. Utworzyć bezpośrednio w kodzie Javy... - słabe
    @GetMapping(path="/time4", produces="text/html")
    @ResponseBody
    public String time4() {
        LocalDateTime dt = LocalDateTime.now();
        return String.format(
                """
                <html><body>
                    <h1>Data i czas</h1>
                    <p>Teraz jest godzina <strong style='color:purple'>%s</strong></p>
                    <p>Dzisiejsza data: <strong style='color:blue'>%s</strong></p>
                    <p style='color: green'>%s</p>
                </body></html>
                """,
                dt.toLocalTime(), dt.toLocalDate(), dt.format(FORMAT_DATY));
    }
    
    // 2. Tworzymy odpowiedź za pomocą "szablonu" z wykorzystaniem dodatkowej technologii,
    // tutaj jest to Thymeleaf, ale może być coś innego: JSP, Velocity, FreeMarker, Mustache, ...
    // https://www.thymeleaf.org/

    // Gdy metoda w kontrolerze w wyniku zwraca String, a nie posiada adnotacji @ResponseBody,
    // to Spring traktuje zwracaną wartość jak nazwę szablonu,
    // w odp. miejscu szuka pliku z szablonem i generuje wynik za pomocą tego szablonu.

    // Gdy do szablonu chcemy przekazać jakieś dane, to dodajemy je jako atrybuty do modelu.
    // Najprościej zrobić to poprzez parametr model.

    @GetMapping("/time5")
    public String time5(Model model) {
        LocalDateTime dt = LocalDateTime.now();
        model.addAttribute("czas", dt);
        return "szablon_czasu_5.html";
    }

    @GetMapping("/time6")
    public String time6(Model model) {
        LocalDateTime dt = LocalDateTime.now();
        model.addAttribute("dt", dt);

        List<LocalDate> dates = new ArrayList<>();
        LocalDate date = dt.toLocalDate();
        for(int i = 0; i <= 5; i++) {
            dates.add(date.plusWeeks(i));
        }
        model.addAttribute("futureWeeks", dates);
        
        return "szablon_czasu_6.html";
    }  
}
