package sklep.rest;

import java.math.BigDecimal;
import java.util.List;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import sklep.db.DBConnection;
import sklep.db.DBException;
import sklep.db.ProductDAO;
import sklep.db.RecordNotFound;
import sklep.model.Product;
import sklep.photo.PhotoUtil;

@Path("/products.json")
// na poziomie klasy można podać domyślne wartości Produces i Consumes dla wszystkich metod
@Produces("application/json")
@Consumes("application/json")
public class RProductJSON {
    @GET
    public List<Product> readAllProducts() throws DBException {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            return productDAO.readAll();
        }
    }
    
    @GET
    @Path("/{id}")
    public Product readOneProduct(@PathParam("id") int productId) throws DBException, RecordNotFound {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            return productDAO.findById(productId);
        }
    }

    @POST
    public void saveProduct(Product product) throws DBException {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            productDAO.save(product);
            db.commit();
        }
    }
    
    @GET
    @Path("/{id}/price")
    public BigDecimal getPrice(@PathParam("id") int productId) throws DBException, RecordNotFound {
        return readOneProduct(productId).getPrice();
    }

    @PUT
    @Path("/{id}/price")
    public void setPrice(@PathParam("id") int productId, BigDecimal newPrice) throws DBException, RecordNotFound {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            Product product = productDAO.findById(productId);
            product.setPrice(newPrice);
            productDAO.save(product);
            db.commit();
        }
    }

    @DELETE
    @Path("/{id}")
    public void usun(@PathParam("id") int productId) throws DBException, RecordNotFound {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            productDAO.delete(productId);
            db.commit();
        }
    }

    // Jeśli na poziomie metody użyjemy adnotacji @Produces lub @Consumes,
    // to ustawienie metody nadpisuje ogólne ustawienie dla klasy
    @GET
    @Path("/{id}/photo")
    @Produces("image/jpeg")
    public byte[] getPhoto(@PathParam("id") int productId) throws DBException, RecordNotFound {
        return PhotoUtil.readBytes(productId);
    }

    @PUT
    @Path("/{id}/photo")
    @Consumes("image/jpeg")
    public void writePhoto(@PathParam("id") int productId, byte[] bajty) {
        PhotoUtil.writeBytes(productId, bajty);
    }
}
