package sklep.soap;

import java.io.ByteArrayInputStream;
import java.math.BigDecimal;
import java.util.List;

import jakarta.jws.WebParam;
import jakarta.jws.WebResult;
import jakarta.jws.WebService;
import jakarta.xml.ws.soap.MTOM;
import sklep.db.DBConnection;
import sklep.db.DBException;
import sklep.db.OrderDAO;
import sklep.db.ProductDAO;
import sklep.db.RecordNotFound;
import sklep.model.Order;
import sklep.model.Product;
import sklep.photo.PhotoUtil;

@WebService
@MTOM
public class Sklep {
    @WebResult(name="product")
    public List<Product> readAllProducts() throws DBException {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            return productDAO.readAll();
        }
    }

    @WebResult(name="product")
    public List<Product> readProductsByPrice(
            @WebParam(name="min") BigDecimal minPrice, 
            @WebParam(name="max") BigDecimal maxPrice) throws DBException {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            return productDAO.findByPrice(minPrice, maxPrice);
        }
    }
    
    @WebResult(name="product")
    public Product readOneProduct(@WebParam(name="id") int productId) throws DBException, RecordNotFound {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            return productDAO.findById(productId);
        }
    }
    
    // zapisywanie produktu (Product jest parametrem wejściowym)
    public void saveProduct(@WebParam(name="product") Product product) throws DBException {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            productDAO.save(product);
            db.commit();
        }
    }
    
    // odczyt zamówienia wg ID a może też odczyt wszystkich zamówień wskazanego klienta
    @WebResult(name="order")
    public Order readOneOrder(@WebParam(name="id") int orderId) throws DBException, RecordNotFound {
        try(DBConnection db = DBConnection.open()) {
            OrderDAO orderDAO = db.orderDAO();
            return orderDAO.findById(orderId);
        }
    }
    
    @WebResult(name="order")
    public List<Order> readCustomerOrders(@WebParam(name="customer-email") String email) throws DBException {
        try(DBConnection db = DBConnection.open()) {
            OrderDAO orderDAO = db.orderDAO();
            return orderDAO.customerOrders(email);
        }
    }
    
    @WebResult(name="bytes")
    public byte[] readPhoto(@WebParam(name="id") int productId) throws DBException, RecordNotFound {
        return PhotoUtil.readBytes(productId);
    }
    
    public void savePhoto(@WebParam(name="id") int productId, @WebParam(name="bytes") byte[] bytes) throws DBException {
        PhotoUtil.writeBytes(productId, bytes);
    }
}
