--* Polecenia modyfikacji danych (DML) - podstawy *--
/* DML - Data Manipulation Language.
   Część standardu SQL odpowiedzialna za pracę z rekordami / z konkretnymi danymi.
   Teoretycznie w skład DML wchodzi również polecenie SELECT.
   W praktyce, gdy potocznie mówimy "DML", to myślimy o poleceniach INSERT UPDATE DELETE

   DDL - Data Definition L. - CREATE, ALTER, DROP
   DCL - GRANT / REVOKE
*/

-- INSERT - wstawia zupełnie nowy rekord do tabeli (takie id nie może wcześniej występować)

INSERT INTO countries(country_id, country_name, region_id)
    VALUES('PL', 'Poland', 10);

-- Gdy podajemy nazwy kolumn do wstawienia, mamy prawo podać je w zmienionej kolejności lub niektóre pominąć.
-- Dla pominiętych przyjmowana jest wartość domyślna z definicji tabeli lub NULL.
-- (Można też jawnie wpisać DEFAULT lub NULL w miejsce wartości.)
INSERT INTO countries(country_name, country_id)
    VALUES('San Escobar', 'Se');

-- Jeśli za tabelą nie podamy nazw kolumn, musimy wstawić wszystkie w tej samej kolejności jak w definicji tabeli.
INSERT INTO countries
    VALUES('CZ', 'Czech Republic', 10);

-- Specyfika Postgresa: w jednym INSERT można podać wiele zestawów wartości, rozdzielając przecinkiem
INSERT INTO countries(country_id, country_name, region_id)
    VALUES    ('CR', 'Croatia', 10)
            , ('SR', 'Serbia', 10);

SELECT * FROM countries ORDER BY country_id;

-- UPDATE - modyfikacja istniejących danych

UPDATE countries SET country_name = 'Czechia' WHERE country_id = 'CZ';

-- Przykład zmiany w wielu rekordach: programistom dajemy podwyżkę 10 tys. i ustawiamy prowizję na 10%
-- Nowa wartość może zależeć od dotychczasowej.
UPDATE employees SET salary = salary + 10000, commission_pct = 0.1 WHERE job_id = 'IT_PROG';

SELECT * FROM employees ORDER BY employee_id;

-- Cofnięcie tych zmian.
-- UPDATE employees SET salary = salary - 10000, comission_pct = NULL WHERE job_id = 'IT_PROG';

UPDATE employees SET salary =
    CASE
    WHEN salary <= 10000
        THEN salary + 5000
        ELSE salary * 1.25
    END;


-- DELETE - usunięcie rekordów wskazanych za pomocą warunku WHERE

DELETE FROM countries WHERE country_id IN ('PL', 'CZ', 'Se', 'CR', 'SR');

-- Uwaga: Brak WHERE spowoduje usunięcie wszystkich rekordów. Uwaga na zaznaczanie.

-- Istnieje też TRUNCATE tabela; -- także usuwa wszystkie rerkody, ale moze działać szybciej
-- Gdyby do tych rekordów odnosiły się rekordy z innych tabel, to TRUCATE tabela CASCADE
-- usunie rekordy także z tych innych tabel.
-- RESTART IDENTITY zrestartuje numerację w kolumnach typu SERIAL


-- Przykład ilustrujący zasadę działania polecenia MERGE.
-- W tabeli rejestr_pracy zbieramy informacje o tym, ile godzin przepracował dany pracownik
-- identyfikowany za pomocą id.
CREATE TABLE rejestr_pracy (
    id INTEGER PRIMARY KEY,
    suma_godzin NUMERIC(6, 1) NOT NULL
);

TRUNCATE rejestr_pracy;

-- Wstawiam pierwsze rekordy.
-- Pracownik nr 1 i 2 istnieją i mają już jakieś godziny
-- Pracowni nr 3 nie istnieje
INSERT INTO rejestr_pracy VALUES (1, 10);
INSERT INTO rejestr_pracy VALUES (2, 10);

SELECT * FROM rejestr_pracy;

-- Za pomocą MERGE będziemy chcieli teraz zwiększyć liczbę godzin o 5 pracownikom 2 i 3.
-- Przy czym pracownik nr 2 już istnieje, i dla mniego trzeba wykonać UPDATE
-- a pracownik nr 3 nie istnieje, i dla niego INSERT
MERGE INTO rejestr_pracy rp
USING (SELECT 2 AS id, 5 AS zmiana
       UNION ALL
       SELECT 3 AS id, 5 AS zmiana) src
ON (rp.id = src.id)
WHEN MATCHED THEN UPDATE
SET suma_godzin = rp.suma_godzin + src.zmiana
WHEN NOT MATCHED THEN INSERT
VALUES (src.id, src.zmiana);

SELECT * FROM rejestr_pracy;

-- Przykład podzapytania w UPDATE.
-- Wykonuję w transakcji, którą później anuluję.
UPDATE employees e
SET salary = salary + (SELECT 10 * count(*) FROM employees WHERE salary >= e.salary)
WHERE job_id = (SELECT job_id FROM jobs WHERE job_title = 'Programmer');

SELECT * FROM employees ORDER BY employee_id;

