package sprzedaz;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;

/* Ta wersja klasy wczytującej dane z pliku stosuje technikę "puli obiektów",
 * czyli w kolejnych rekordach wykorzystuje jako wartości pól te same obiekty,
 * które zostały użyte wcześniej, jeśli wartości pól są jednakowe.
 * Dla klasy `String` korzystam z istniejącej metody `intern`, która
 * jest realizowana bezpośrednio przez JVM. Natomaist dla klas `BigDecimal` oraz `LocalDate`
 * tworzę własną implementację tego mechanizmu opartą o słowniki.
 * Ta technika służy przede wszystkim oszczędzaniu pamięci,
 * może zwrócić się czasowo, jeśli dane po wczytaniu będą później intensywnie używane chociażby w porównaniach,
 * jest poprawna w przypadku klas niemutowalnych.
 */

public class MaszynaWczytujaca2 {
	private static final String DOMYSLNY_PLIK = "sprzedaz.csv";
    private final Map<String, LocalDate> pulaDat = new ConcurrentHashMap<>();
	private final Map<String, BigDecimal> pulaCen = new ConcurrentHashMap<>();

	public Transakcja wczytajJedenRekord(String linia) {
		String[] pola = linia.split(",");
		int sztuk = Integer.parseInt(pola[6]);

        // "2020-03-04" → LocalDate(2020, 3, 4)
		LocalDate data = pulaDat.computeIfAbsent(pola[0], LocalDate::parse);
		BigDecimal cena = pulaCen.computeIfAbsent(pola[5], BigDecimal::new);

		String miasto = pola[1].intern();
		String sklep = pola[2].intern();
		String kategoria = pola[3].intern();
		String towar = pola[4].intern();

		return new Transakcja(data, miasto, sklep, kategoria, towar, cena, sztuk);
	}

	public static List<Transakcja> wczytajCalyPlik(File plik) {
		MaszynaWczytujaca2 maszynaWczytujaca = new MaszynaWczytujaca2();

		List<Transakcja> lista = new ArrayList<>();
		try(BufferedReader reader = Files.newBufferedReader(plik.toPath())) {
			String linia = reader.readLine(); // ignoruję nagłówki
			while((linia = reader.readLine()) != null) {
				Transakcja transakcja = maszynaWczytujaca.wczytajJedenRekord(linia);
				lista.add(transakcja);
			}
			return lista;
		} catch (IOException e) {
            throw new RuntimeException(e);
		}
	}

    public static List<Transakcja> wczytajCalyPlik(String plik) {
        return wczytajCalyPlik(new File(plik));
    }

    public static List<Transakcja> wczytajCalyPlik() throws IOException {
        return wczytajCalyPlik(DOMYSLNY_PLIK);
    }

    public static Stream<Transakcja> dajStrumien(Path plik, boolean parallel) {
        try {
            MaszynaWczytujaca2 maszynaWczytujaca = new MaszynaWczytujaca2();
            Stream<String> s = Files.lines(plik).skip(1);
            return (parallel ? s.parallel() : s)
                    .map(maszynaWczytujaca::wczytajJedenRekord);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static Stream<Transakcja> dajStrumien(String plik, boolean parallel) {
        return dajStrumien(Paths.get(plik), parallel);
    }

    public static Stream<Transakcja> dajStrumien(boolean parallel) {
        return dajStrumien(DOMYSLNY_PLIK, parallel);
    }

}
