package sklep;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.LinkedHashMap;
import java.util.Map;

public class Towar {
	private static final String SEP = ";";
	private final String nazwa;
	private BigDecimal cenaBrutto;
	private BigDecimal vat;
	private int stan;
	
	public Towar(String nazwa, BigDecimal cenaBrutto, BigDecimal vat, int stan) {
		this.nazwa = nazwa;
		this.cenaBrutto = cenaBrutto;
		this.vat = vat;
		this.stan = stan;
	}

	public String getNazwa() {
		return nazwa;
	}

	public BigDecimal getCenaBrutto() {
		return cenaBrutto;
	}

	public BigDecimal getVat() {
		return vat;
	}

	public int getStan() {
		return stan;
	}

	@Override
	public String toString() {
		return "Towar [nazwa=" + nazwa + ", cenaBrutto=" + cenaBrutto + ", vat=" + vat + ", stan=" + stan + "]";
	}
	
	public BigDecimal getCenaNetto() {
		// return cenaBrutto / (1 + vat);
		return cenaBrutto.divide(BigDecimal.ONE.add(vat), 2, RoundingMode.HALF_UP);
	}
	
	public void zmiejszStan(int ileSztuk) {
		if(ileSztuk <= 0) {
			throw new IllegalArgumentException("Ujemna liczba");
		}
		if(ileSztuk > this.stan) {
			throw new IllegalArgumentException("Za duża liczba w parametrze");
		}
		this.stan -= ileSztuk;
	}
	
	// W tym przykładzie ogólne operacje dotyczące towarów, jak odczyt z pliku i zapis do pliku
	// umieścimy w samej klasie Towar jako metody statyczne.
	
	/** Odczytuje dane towarów z pliku z formacie CSV i zwraca w formie słownika indeksowanego nazwami produktów.
	 * 
	 * @param sciezka ścieżka do pliku CSV
	 * @return słownik, gdzie kluczami są nazwy towarów, a wartościami są całe obiekty Towar
	 */
	public static Map<String, Towar> wczytajPlik(String sciezka) {
		Map<String, Towar> towary = new LinkedHashMap<>();
		
		try(BufferedReader in = Files.newBufferedReader(Paths.get(sciezka))) {
			String linia;
			while((linia = in.readLine()) != null) {
				String[] t = linia.split(SEP);
				Towar towar = new Towar(t[0], new BigDecimal(t[1]), new BigDecimal(t[2]).movePointLeft(2), Integer.parseInt(t[3]));
				towary.put(t[0], towar);
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		return towary;
	}

	public static void zapiszPlik(Map<String, Towar> towary, String sciezka) {
		try(PrintWriter out = new PrintWriter(sciezka)) {
			for(Towar towar : towary.values()) {
				out.print(towar.getNazwa());
				out.print(SEP);
				out.print(towar.getCenaBrutto());
				out.print(SEP);
				out.print(towar.getVat().movePointRight(2));
				out.print(SEP);
				out.print(towar.getStan());
				out.println();
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}
}
