package sklep.controller;

import jakarta.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.*;
import sklep.basket.Basket;
import sklep.model.Product;
import sklep.photo.PhotoUtil;
import sklep.repository.ProductRepository;

import java.math.BigDecimal;
import java.util.List;
import java.util.Optional;

@Controller
@RequestMapping("/products")
public class ProductController {
    @Autowired
    private ProductRepository productRepository;

    @Autowired
    private PhotoUtil photoUtil;

    @GetMapping
    public String odczytajProdukty(Model model) {
        List<Product> products = productRepository.findAll(Sort.by("id"));
        model.addAttribute("products", products);
        return "products";
    }

    @GetMapping("/{id}")
    public String odczytajJedenProdukt(@PathVariable int id, Model model) {
        Optional<Product> product = productRepository.findById(id);
        if(product.isPresent()) {
            model.addAttribute("product", product.get());
            return "product";
        } else {
            model.addAttribute("product_id", id);
            return "missing_product";
        }
    }

    @GetMapping("/szukaj")
    public String szukaj(Model model,
                         @RequestParam(required=false) String name,
                         @RequestParam(required=false) BigDecimal min,
                         @RequestParam(required=false) BigDecimal max) {
        List<Product> products = List.of();
        if(name != null && !name.isEmpty() && min == null && max == null) {
            products = productRepository.findByProductNameContainsIgnoringCase(name);
        } else if ((name == null || name.isEmpty()) && (min != null || max != null)) {
            if(min == null) {
                min = BigDecimal.ZERO;
            }
            if(max == null) {
                max = BigDecimal.valueOf(1000_000_000);
            }
            products = productRepository.findByPriceBetween(min, max);
        } else if (name != null && !name.isEmpty()) {
            if(min == null) {
                min = BigDecimal.ZERO;
            }
            if(max == null) {
                max = BigDecimal.valueOf(1000_000_000);
            }
            products = productRepository.findByProductNameContainingIgnoringCaseAndPriceBetween(name, min, max);
        }
        model.addAttribute("products", products);
        return "wyszukiwarka2";
    }

    @GetMapping("/{id}/add-to-basket")
    public String addToBasket(@PathVariable("id") Integer productId,
                              @SessionAttribute Basket basket) {
        Optional<Product> product = productRepository.findById(productId);
        if(product.isPresent()) {
            basket.addProduct(product.get());
        } else {
            System.err.println("Nieznany produkt dodawany do koszyka: " + productId);
        }
        return "redirect:/products";
    }

    @GetMapping("/new")
    public String nowyProdukt(@ModelAttribute Product product) {
        return "product_form";
    }

    @GetMapping("/{id}/edit")
    public String edytujProdukt(@PathVariable int id, Model model) {
        Optional<Product> product = productRepository.findById(id);
        if(product.isPresent()) {
            model.addAttribute("product", product.get());
            return "product_form";
        } else {
            model.addAttribute("product_id", id);
            return "missing_product";
        }
    }

    @PostMapping({"/new", "/{id}/edit"})
    // Ta metoda zapisuje dane przysłane z formularza obojętnie, czy to było edit, czy new
    // Adnotacja @Valid powoduje, że Spring dokona walidacji obiektu PRZED uruchomieniem tej metody.
    // Jeśli nie ma dodatkowego parametru BindingResult, a są błędy walidacji, to Spring naszej metody w ogóle nie uruchomi.
    // Jeśli jednak metoda ma parametr BindingResult, to
    // metoda zawsze jest uruchamiana przez Springa,
    // a w tym parametrze zawarte są informacje o przebiegu walidacji, w tym błędy.
    // Bez adnotacji @ModelAttribute też działało
    public String saveProduct(@ModelAttribute @Valid Product product,
                              BindingResult bindingResult) {
        // W tej wersji dane z wypełnionego formularza odbieramy w postaci jednego obiektu Product.
        // Spring sam wpisze dane do pól o takich samych nazwach.
        // Taki parametr od razu staje się częścią modelu (to jest tzw. ModelAttribute)

        if(bindingResult.hasErrors()) {
            System.err.println("Są błędy: " + bindingResult.getAllErrors());
            // normalnie wyświetlilibyśmy coś na stronie...
            // ale robi to za nas tag f:form i f:errors
            // Dopóki są błędy, pozostajemy na stronie formularza.
            return "product_form";
        } else {
            // Gdy podczas próby zapisu (operacja save) obiekt nie spełnia warunków walidacji, to jest wyrzucany wyjątek.
            System.out.println("id przed zapisem: " + product.getId());
            productRepository.save(product);
            System.out.println("id po zapisie: " + product.getId());
            return "redirect:/products";
            // Po zaakceptowaniu danych, klient jest przenoszony do listy produktów.
        }
    }

    @GetMapping(path="/{id}/photo", produces="image/jpeg")
    @ResponseBody
    public byte[] getPhoto(@PathVariable int id) {
        return photoUtil.readBytes(id);
    }
}
