package sklep.rest;

import jakarta.ws.rs.*;
import sklep.db.DBConnection;
import sklep.db.DBException;
import sklep.db.ProductDAO;
import sklep.db.RecordNotFound;
import sklep.model.Product;

import java.math.BigDecimal;
import java.util.List;

@Path("/products.json")
@Produces("application/json")
@Consumes("application/json")
public class RProductsJson {
    @GET
    public List<Product> allProducts() throws DBException {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            return productDAO.readAll();
        }
    }

    @GET
    @Path("/{id}")
    // przykładowo /api/products.json/3
    public Product oneProduct(@PathParam("id") int productId) throws DBException, RecordNotFound {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            return productDAO.findById(productId);
        }
    }

    @POST
    public void saveProduct(Product product) throws DBException {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            productDAO.save(product);
            db.commit();
        }
    }

    // Nie praktykuje się tego zbyt często, ale można zdefiniować dedykowane metody dające dostęp do poszczególnych pól obiektu (aby nie transferować całego rekordu, gdy potrzebna tylko jedna informacja)
    @GET
    @Path("/{id}/price")
    public BigDecimal getPrice(@PathParam("id") int productId) throws DBException, RecordNotFound {
        return oneProduct(productId).getPrice();
    }

    @PUT
    @Path("/{id}/price")
    public void setPrice(@PathParam("id") int productId, BigDecimal newPrice) throws DBException, RecordNotFound {
        try(DBConnection db = DBConnection.open()) {
            ProductDAO productDAO = db.productDAO();
            Product product = productDAO.findById(productId);
            product.setPrice(newPrice);
            productDAO.save(product);
            db.commit();
        }
    }

    @GET
    @Path("/{id}/photo")
    @Produces("image/jpeg")
    public byte[] getPhoto(@PathParam("id") int productId) throws DBException, RecordNotFound {
        return PhotoUtil.readBytes(productId);
    }

    @PUT
    @Path("/{id}/photo")
    @Consumes("image/jpeg")
    public void getPhoto(@PathParam("id") int productId, byte[] bajty) {
        PhotoUtil.writeBytes(productId, bajty);
    }
}
